#' Calculate alignment coverage
#'
#' This function calculates alignment coverage
#' according to the read strand and length from alignment results.
#'
#' This function calculates alignment coverage from the two BAM files,
#' `*.clean.t1.bam` and `*.clean.t2.bam`,
#' generated by the \code{\link{align_reads}} function.
#' The coverage is then sorted by the strand and length of the aligned reads
#' and summarized into data frames.
#'
#' @param x A \code{\link{CircSeqAlignTkAlign-class}} object generated by
#'     the \code{\link{align_reads}} function.
#' @return A \code{\link{CircSeqAlignTkCoverage-class}} object.
#' @examples
#' output_dpath <- tempdir()
#' genome_seq <- system.file(package="CircSeqAlignTk", "extdata", "FR851463.fa")
#' fq <- system.file(package="CircSeqAlignTk", "extdata", "srna.fq.gz")
#' ref_index <- build_index(input = genome_seq,
#'                          output = file.path(output_dpath, 'index'))
#' aln <- align_reads(input = fq, index = ref_index,
#'                    output = file.path(output_dpath, 'align_results'))
#' 
#' alncov <- calc_coverage(aln)
#' @seealso \code{\link{CircSeqAlignTkAlign-class}},
#'     \code{\link{CircSeqAlignTkCoverage-class}}, \code{\link{align_reads}}
#' @export
calc_coverage <- function(x) {
    depth_mat_1 <- .calc_coverage(x@clean_bam[1], x@reference@length)
    depth_mat_2 <- .calc_coverage(x@clean_bam[2], x@reference@length,
                                  x@reference@cut_loc)

    read_len <- sort(unique(c(colnames(depth_mat_1[[1]]),
                              colnames(depth_mat_1[[2]]),
                              colnames(depth_mat_2[[1]]),
                              colnames(depth_mat_2[[2]]))))

    mat_fwd <- matrix(0, ncol = length(read_len), nrow = x@reference@length)
    mat_rev <- matrix(0, ncol = length(read_len), nrow = x@reference@length)
    colnames(mat_fwd) <- colnames(mat_rev) <- read_len

    .add_coverage <- function(m0, m1, cname) {
        if (!any(is.na(m1))) {
            idx <- (colnames(m1) %in% cname)
            if (any(idx)) m0[, cname] <- m0[, cname] + m1[, idx]
        }
        m0
    }
    for (cname in read_len) {
        mat_fwd <- .add_coverage(mat_fwd, depth_mat_1[[1]], cname)
        mat_fwd <- .add_coverage(mat_fwd, depth_mat_2[[1]], cname)
        mat_rev <- .add_coverage(mat_rev, depth_mat_1[[2]], cname)
        mat_rev <- .add_coverage(mat_rev, depth_mat_2[[2]], cname)
    }

    new('CircSeqAlignTkCoverage',
        forward = mat_fwd,
        reverse = mat_rev,
        .figdata = rbind(reshape_coverage_df(mat_fwd, '+'),
                         reshape_coverage_df(mat_rev, '-')))
}


## ' @importFrom methods setGeneric
# setGeneric('coverage', signature = 'x', function(x, ...) standardGeneric('coverage'))

## ' @importFrom methods setMethod
## ' @rdname calc_coverage
## ' @export
# setMethod('coverage', 'CircSeqAlignTkCoverage',
#           function(x, ...) {
#               calc_coverage(x, ...)
#           }
# )




#' @importFrom IRanges IRanges coverage
#' @importFrom Rsamtools ScanBamParam scanBam
.calc_coverage <- function(bam_fpath, ref_length, pos_shift = 0) {
    if (!file.exists(bam_fpath)) return(list(NA, NA))

    scan_bam_params <- ScanBamParam(tag = c('NM'),
                                    what = c('qwidth', 'flag', 'pos'))
    bam <- scanBam(bam_fpath, param = scan_bam_params)[[1]]

    read_length <- sort(unique(bam$qwidth))
    if (length(read_length) == 0) return(list(NA, NA))

    depth_mat_fwd <- matrix(0, nrow = ref_length, ncol = length(read_length))
    depth_mat_rev <- matrix(0, nrow = ref_length, ncol = length(read_length))
    colnames(depth_mat_fwd) <- paste0('L', read_length)
    colnames(depth_mat_rev) <- paste0('L', read_length)
    for (i in seq(length(read_length))) {
        # forward strand
        is_target <- ((bam$qwidth == read_length[i]) & (bam$flag == 0))
        depth_mat_fwd[, i] <- as.integer(
                                coverage(IRanges(bam$pos[is_target],
                                                 width = bam$qwidth[is_target]),
                                         width = ref_length))
        # reverse strand
        is_target <- ((bam$qwidth == read_length[i]) & (bam$flag == 16))
        depth_mat_rev[, i] <- as.integer(
                                coverage(IRanges(bam$pos[is_target],
                                                 width = bam$qwidth[is_target]),
                                         width = ref_length))
    }

    if (pos_shift > 0) {
        depth_mat_fwd <- rbind(
            depth_mat_fwd[seq((ref_length - pos_shift + 1 + 1), ref_length), ],
            depth_mat_fwd[seq(1, (ref_length - pos_shift + 1)), ])
        depth_mat_rev <- rbind(
            depth_mat_rev[seq((ref_length - pos_shift + 1 + 1), ref_length), ],
            depth_mat_rev[seq(1, (ref_length - pos_shift + 1)), ])
    }

    list(depth_mat_fwd, depth_mat_rev)
}




#' Visualize alignment coverage
#'
#' This function visualizes the alignment coverage using an area chart.
#' By default, the upper and lower directions of the y-axis represent
#' the alignment coverage of the reads
#' aligned in the forward and reverse strands, respectively.
#'
#' @param x A \code{\link{CircSeqAlignTkCoverage-class}} object generated by
#'     the \code{\link{calc_coverage}} function.
#' @param read_lengths Numeric numbers to specify the lengths of reads
#'     targeted for visualization. If `NULL` (default),
#'     plot the alignment coverage of reads with all lengths.
#' @param fill Specify `NULL` or `read_length`.
#'     If `read_length` is specified, then color the area chart
#'             according to the read length.
#' @param scale_fun Set `log10` or `log` to plot the alignment coverage
#'     in logarithmic scale.
#' @param ... Other graphical parameters.
#' @return An object of ggplot2.
#' @examples
#' output_dpath <- tempdir()
#' genome_seq <- system.file(package="CircSeqAlignTk", "extdata", "FR851463.fa")
#' fq <- system.file(package="CircSeqAlignTk", "extdata", "srna.fq.gz")
#' ref_index <- build_index(input = genome_seq,
#'                          output = file.path(output_dpath, 'index'))
#' aln <- align_reads(input = fq, index = ref_index,
#'                    output = file.path(output_dpath, 'align_results'))
#' 
#' alncov <- calc_coverage(aln)
#' plot(alncov)
#' @seealso \code{\link{CircSeqAlignTkCoverage-class}}, \code{\link{calc_coverage}}
#' @importFrom magrittr %>%
#' @importFrom dplyr mutate group_by summarise filter
#' @importFrom ggplot2 ggplot geom_area aes aes_string
#' @importFrom ggplot2 xlim scale_y_continuous guides guide_legend
#' @export
plot_coverage <- function(x, read_lengths = NULL, fill = 'read_length', scale_fun = NULL) {
    x <- x@.figdata
    if (!is.null(read_lengths))
        x <- x %>% filter(.data$read_length %in% read_lengths)
    x_fwd <- x %>% filter(.data$strand == '+')
    x_rev <- x %>% filter(.data$strand == '-')

    if (is.null(fill)) {
        x_fwd <- x_fwd %>% group_by(.data$position) %>%
            summarise(coverage = sum(.data$coverage)) %>%
            mutate(strand = '+')
        x_rev <- x_rev %>% group_by(.data$position) %>%
            summarise(coverage = sum(.data$coverage)) %>%
            mutate(strand = '-')
    }
    x_fwd$strand <- factor(x_fwd$strand, levels = c('+', '-'))
    x_rev$strand <- factor(x_rev$strand, levels = c('+', '-'))
    x_fwd$coverage <- .scale_coverage(x_fwd$coverage, scale_fun)
    x_rev$coverage <- .scale_coverage(x_rev$coverage, scale_fun)

    ggplot() +
        geom_area(data = x_fwd,
                  aes_string(x = 'position', y = 'coverage', fill = fill),
                  stat = 'identity', position = 'stack') +
        geom_area(data = x_rev,
                  aes_string(x = 'position', y = '- coverage', fill = fill),
                  stat = 'identity', position = 'stack') +
        xlim(range(x$position)) +
        scale_y_continuous(labels = function(x) abs(x)) +
        guides(fill = guide_legend(title = 'read length'))
}

.scale_coverage <- function(x, scale_fun) {
    if (!is.null(scale_fun)) {
        if (is(scale_fun, 'function')) {
            scale_base <- rep(TRUE, length = length(x))
            if (identical(scale_fun, log10)) scale_base <- (x >= 10)
            if (identical(scale_fun, log)) scale_base <- (x >= exp(1))
            x[scale_base] <- scale_fun(x[scale_base])
            x[!scale_base] <- x[!scale_base] / 10
        } else {
            stop('Set a function to `scale_fun` argument.')
        }
    }
    x
}



#' @rdname plot_coverage
#' @export
plot.CircSeqAlignTkCoverage <- function(x, ...) {
    plot_coverage(x, ...)
}


