## ----setup, include=FALSE-----------------------------------------------------
knitr::opts_chunk$set(echo = TRUE)
library(knitr)
library(kableExtra)

## ----example_data-------------------------------------------------------------
ibed1_file <- system.file("extdata", "ibed1_example.zip", package = "HiCaptuRe")
ibed2_file <- system.file("extdata", "ibed2_example.zip", package = "HiCaptuRe")
peakmatrix_file <- system.file("extdata", "peakmatrix_example.zip", package = "HiCaptuRe")
annotation_file <- system.file("extdata", "annotation_example.txt", package = "HiCaptuRe")

## ----library, comment=F, message=F--------------------------------------------
library(HiCaptuRe)

## ----load_interactions--------------------------------------------------------
ibed1 <- load_interactions(file = ibed1_file, genome = "BSgenome.Hsapiens.NCBI.GRCh38")

## ----ibed1--------------------------------------------------------------------
ibed1

## ----slots--------------------------------------------------------------------
slotNames(ibed1)

## ----parameters---------------------------------------------------------------
getParameters(x = ibed1)

## ----enzymes table,echo=FALSE-------------------------------------------------
enzymes <- data.frame(
    Enzyme = c("HindIII", "EcoRI", "BamHI", "MboI", "DpnII"),
    Motif = c("A^AGCTT", "G^AATTC", "G^GATCC", "^GATC", "^GATC"),
    Cut_Position = c(1, 1, 1, 0, 0)
)

kable(enzymes, caption = "Table: Recognized Restriction Enzymes and Their Motifs") |>
    kable_styling(
        bootstrap_options = c("striped", "hover", "condensed", "responsive"),
        full_width = FALSE, position = "center"
    ) |>
    row_spec(0, extra_css = "text-align: center;")

## ----digest_genome------------------------------------------------------------
digest <- digest_genome(genome = "BSgenome.Hsapiens.NCBI.GRCh38", RE_name = "HindIII")
head(digest$digest)

## ----annotate-----------------------------------------------------------------
ibed1_annotated <- annotate_interactions(
    interactions = ibed1,
    annotation = annotation_file
)
ibed1_annotated

## ----annotate_parameters------------------------------------------------------
parameters <- getParameters(x = ibed1_annotated)
parameters$annotate

## ----interactionsByBaits------------------------------------------------------
baits_of_interest <- c("DAZAP1", "PLIN3", "FPR3", "TP53")

ibed_byBaits <- interactionsByBaits(
    interactions = ibed1_annotated,
    baits = baits_of_interest
)

ibed_byBaits

## ----ByBaits------------------------------------------------------------------
getByBaits(ibed_byBaits)

## ----interactionsByRegions----------------------------------------------------
regions <- GenomicRanges::GRanges(
    seqnames = 19,
    ranges = IRanges(start = c(500000, 1000000), end = c(510000, 1100000))
)

ibed_byRegions <- interactionsByRegions(
    interactions = ibed1_annotated,
    regions = regions
)

ibed_byRegions

## ----byRegions----------------------------------------------------------------
getByRegions(ibed_byRegions)

## ----intersect_interactions---------------------------------------------------
ibed2 <- load_interactions(file = ibed2_file, genome = "BSgenome.Hsapiens.NCBI.GRCh38")
ibed2_annotated <- annotate_interactions(interactions = ibed2, annotation = annotation_file)

interactions_list <- list(A = ibed1_annotated, B = ibed2_annotated)

output <- intersect_interactions(interactions_list = interactions_list)

## ----intersections------------------------------------------------------------
lapply(output$intersections, function(x) x[1:2])

## ----upset--------------------------------------------------------------------
output$upset

## ----venn---------------------------------------------------------------------
output$venn

## ----distance_summary---------------------------------------------------------
dist_sum <- distance_summary(
    interactions = ibed1_annotated,
    breaks = seq(0, 10^6, 10^5),
    sample = "ibed1"
)
dist_sum

## ----absolute, fig.show="hold", out.width="50%"-------------------------------
plots <- plot_distance_summary(distances = dist_sum, type_of_value = "absolute")
plots$int_dist
plots$total_dist

## ----by_int_type--------------------------------------------------------------
plots <- plot_distance_summary(distances = dist_sum, type_of_value = "by_int_type")
plots$int_dist_norm_int

## ----by_total-----------------------------------------------------------------
plots <- plot_distance_summary(distances = dist_sum, type_of_value = "by_total")
plots$int_dist_norm_total
plots$total_dist_norm_total

## ----peakmatrix2list----------------------------------------------------------
peakmatrix <- load_interactions(peakmatrix_file, genome = "BSgenome.Hsapiens.NCBI.GRCh38")
interactions_list <- peakmatrix2list(peakmatrix = peakmatrix)
names(interactions_list)

## ----export, eval=F-----------------------------------------------------------
# export_interactions(
#     interactions = ibed1_annotated,
#     file = "/path/to/folder/ibed_annotated.ibed",
#     type = "ibed"
# )

## ----sessioninfo--------------------------------------------------------------
sessionInfo()

