//
// Channel.h
//
// Library: Foundation
// Package: Logging
// Module:  Channel
//
// Definition of the Channel class.
//
// Copyright (c) 2004-2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// SPDX-License-Identifier:	BSL-1.0
//


#ifndef Foundation_Channel_INCLUDED
#define Foundation_Channel_INCLUDED


#include "Poco/Foundation.h"
#include "Poco/Configurable.h"
#include "Poco/Mutex.h"
#include "Poco/RefCountedObject.h"
#include "Poco/AutoPtr.h"


namespace Poco {


class Message;


class Foundation_API Channel: public Configurable, public RefCountedObject
	/// The base class for all Channel classes.
	///
	/// Supports reference counting based garbage
	/// collection and provides trivial implementations
	/// of getProperty() and setProperty().
{
public:
	using Ptr = AutoPtr<Channel>;

	Channel();
		/// Creates the channel and initializes
		/// the reference count to one.

	virtual void open();
		/// Does whatever is necessary to open the channel.
		/// The default implementation does nothing.

	virtual void close();
		/// Does whatever is necessary to close the channel.
		/// The default implementation does nothing.

	virtual void log(const Message& msg) = 0;
		/// Logs the given message to the channel. Must be
		/// overridden by subclasses.
		///
		/// If the channel has not been opened yet, the log()
		/// method will open it.

	virtual void log(Message&& msg);
		/// Logs the given message to the channel by moving it.
		///
		/// This overload allows channels that need to store or queue
		/// messages (such as AsyncChannel) to avoid copying the message.
		///
		/// The default implementation simply calls log(const Message&),
		/// which is appropriate for channels that immediately process
		/// and discard the message (e.g., ConsoleChannel, FileChannel).
		///
		/// Subclasses that buffer or queue messages should override
		/// this method to take advantage of move semantics.

	void setProperty(const std::string& name, const std::string& value) override;
		/// Throws a PropertyNotSupportedException.

	std::string getProperty(const std::string& name) const override;
		/// Throws a PropertyNotSupportedException.

protected:
	~Channel() override;

private:
	Channel(const Channel&);
	Channel& operator = (const Channel&);
};


} // namespace Poco


#endif // Foundation_Channel_INCLUDED
