/*
 * Decompiled with CFR 0.152.
 */
package ai.djl.util.cuda;

import ai.djl.Device;
import ai.djl.engine.EngineException;
import ai.djl.util.Utils;
import ai.djl.util.cuda.CudaLibrary;
import com.sun.jna.Native;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.lang.management.MemoryUsage;
import java.util.ArrayList;
import java.util.Locale;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class CudaUtils {
    private static final Logger logger = LoggerFactory.getLogger(CudaUtils.class);
    private static final CudaLibrary LIB = CudaUtils.loadLibrary();
    private static String[] gpuInfo;
    private static boolean logging;

    private CudaUtils() {
    }

    public static boolean hasCuda() {
        return CudaUtils.getGpuCount() > 0;
    }

    public static int getGpuCount() {
        if (Boolean.getBoolean("ai.djl.util.cuda.fork")) {
            if (gpuInfo == null) {
                gpuInfo = CudaUtils.execute(-1);
            }
            try {
                return Integer.parseInt(gpuInfo[0]);
            }
            catch (NumberFormatException e) {
                logger.warn("Unexpected output: {}", (Object)gpuInfo[0], (Object)e);
                return 0;
            }
        }
        if (LIB == null) {
            return 0;
        }
        int[] count = new int[1];
        int result = LIB.cudaGetDeviceCount(count);
        switch (result) {
            case 0: {
                return count[0];
            }
            case 100: {
                if (logging) {
                    logger.debug("No GPU device found: {} ({})", (Object)LIB.cudaGetErrorString(result), (Object)result);
                }
                return 0;
            }
        }
        if (logging) {
            logger.warn("Failed to detect GPU count: {} ({})", (Object)LIB.cudaGetErrorString(result), (Object)result);
        }
        return 0;
    }

    public static int getCudaVersion() {
        if (Boolean.getBoolean("ai.djl.util.cuda.fork")) {
            int version;
            if (gpuInfo == null) {
                gpuInfo = CudaUtils.execute(-1);
            }
            if ((version = Integer.parseInt(gpuInfo[1])) == -1) {
                throw new IllegalArgumentException("No cuda device found.");
            }
            return version;
        }
        if (LIB == null) {
            throw new IllegalStateException("No cuda library is loaded.");
        }
        int[] version = new int[1];
        int result = LIB.cudaRuntimeGetVersion(version);
        CudaUtils.checkCall(result);
        return version[0];
    }

    public static String getCudaVersionString() {
        int version = CudaUtils.getCudaVersion();
        int major = version / 1000;
        int minor = version / 10 % 10;
        return String.format(Locale.ROOT, "%02d", major) + minor;
    }

    public static String getComputeCapability(int device) {
        if (Boolean.getBoolean("ai.djl.util.cuda.fork")) {
            if (gpuInfo == null) {
                gpuInfo = CudaUtils.execute(-1);
            }
            if (device >= gpuInfo.length - 2) {
                throw new IllegalArgumentException("Invalid device: " + device);
            }
            return gpuInfo[device + 2];
        }
        if (LIB == null) {
            throw new IllegalStateException("No cuda library is loaded.");
        }
        int attrComputeCapabilityMajor = 75;
        int attrComputeCapabilityMinor = 76;
        int[] major = new int[1];
        int[] minor = new int[1];
        CudaUtils.checkCall(LIB.cudaDeviceGetAttribute(major, attrComputeCapabilityMajor, device));
        CudaUtils.checkCall(LIB.cudaDeviceGetAttribute(minor, attrComputeCapabilityMinor, device));
        return String.valueOf(major[0]) + minor[0];
    }

    public static MemoryUsage getGpuMemory(Device device) {
        if (!device.isGpu()) {
            throw new IllegalArgumentException("Only GPU device is allowed.");
        }
        if (Boolean.getBoolean("ai.djl.util.cuda.fork")) {
            String[] ret = CudaUtils.execute(device.getDeviceId());
            if (ret.length != 3) {
                throw new IllegalArgumentException(ret[0]);
            }
            long total = Long.parseLong(ret[1]);
            long used = Long.parseLong(ret[2]);
            return new MemoryUsage(-1L, used, used, total);
        }
        if (LIB == null) {
            throw new IllegalStateException("No GPU device detected.");
        }
        int[] currentDevice = new int[1];
        CudaUtils.checkCall(LIB.cudaGetDevice(currentDevice));
        CudaUtils.checkCall(LIB.cudaSetDevice(device.getDeviceId()));
        long[] free = new long[1];
        long[] total = new long[1];
        CudaUtils.checkCall(LIB.cudaMemGetInfo(free, total));
        CudaUtils.checkCall(LIB.cudaSetDevice(currentDevice[0]));
        long committed = total[0] - free[0];
        return new MemoryUsage(-1L, committed, committed, total[0]);
    }

    public static void main(String[] args) {
        logging = false;
        int gpuCount = CudaUtils.getGpuCount();
        if (args.length == 0) {
            if (gpuCount <= 0) {
                System.out.println("0,-1");
                return;
            }
            int cudaVersion = CudaUtils.getCudaVersion();
            StringBuilder sb = new StringBuilder();
            sb.append(gpuCount).append(',').append(cudaVersion);
            for (int i = 0; i < gpuCount; ++i) {
                sb.append(',').append(CudaUtils.getComputeCapability(i));
            }
            System.out.println(sb);
            return;
        }
        try {
            int deviceId = Integer.parseInt(args[0]);
            if (deviceId < 0 || deviceId >= gpuCount) {
                System.out.println("Invalid device: " + deviceId);
                return;
            }
            MemoryUsage mem = CudaUtils.getGpuMemory(Device.gpu(deviceId));
            String cc = CudaUtils.getComputeCapability(deviceId);
            System.out.println(cc + ',' + mem.getMax() + ',' + mem.getUsed());
        }
        catch (NumberFormatException e) {
            System.out.println("Invalid device: " + args[0]);
        }
    }

    private static CudaLibrary loadLibrary() {
        block12: {
            try {
                if (Boolean.getBoolean("ai.djl.util.cuda.fork")) {
                    return null;
                }
                if (System.getProperty("os.name").startsWith("Win")) {
                    String path = Utils.getenv("PATH");
                    if (path == null) {
                        return null;
                    }
                    Pattern p = Pattern.compile("cudart64_\\d+\\.dll");
                    String cudaPath = Utils.getenv("CUDA_PATH");
                    String[] searchPath = cudaPath == null ? path.split(";") : (cudaPath + "\\bin\\;" + path).split(";");
                    for (String item : searchPath) {
                        File dir = new File(item);
                        File[] files = dir.listFiles(n -> p.matcher(n.getName()).matches());
                        if (files == null || files.length <= 0) continue;
                        String fileName = files[0].getName();
                        String cudaRt = fileName.substring(0, fileName.length() - 4);
                        if (logging) {
                            logger.debug("Found cudart: {}", (Object)files[0].getAbsolutePath());
                        }
                        return (CudaLibrary)Native.load((String)cudaRt, CudaLibrary.class);
                    }
                    if (logging) {
                        logger.debug("No cudart library found in path.");
                    }
                    return null;
                }
                return (CudaLibrary)Native.load((String)"cudart", CudaLibrary.class);
            }
            catch (UnsatisfiedLinkError e) {
                if (logging) {
                    logger.debug("cudart library not found.");
                    logger.trace("", (Throwable)e);
                }
            }
            catch (LinkageError e) {
                if (logging) {
                    logger.warn("You have a conflict version of JNA in the classpath.");
                    logger.debug("", (Throwable)e);
                }
            }
            catch (SecurityException e) {
                if (!logging) break block12;
                logger.warn("Access denied during loading cudart library.");
                logger.trace("", (Throwable)e);
            }
        }
        return null;
    }

    private static String[] execute(int deviceId) {
        String[] stringArray;
        block11: {
            String javaHome = System.getProperty("java.home");
            String classPath = System.getProperty("java.class.path");
            String os = System.getProperty("os.name");
            ArrayList<String> cmd = new ArrayList<String>(4);
            if (os.startsWith("Win")) {
                cmd.add(javaHome + "\\bin\\java.exe");
            } else {
                cmd.add(javaHome + "/bin/java");
            }
            cmd.add("-cp");
            cmd.add(classPath);
            cmd.add("ai.djl.util.cuda.CudaUtils");
            if (deviceId >= 0) {
                cmd.add(String.valueOf(deviceId));
            }
            Process ps = new ProcessBuilder(cmd).redirectErrorStream(true).start();
            InputStream is = ps.getInputStream();
            try {
                String line = Utils.toString(is).trim();
                stringArray = line.split(",");
                if (is == null) break block11;
            }
            catch (Throwable throwable) {
                try {
                    if (is != null) {
                        try {
                            is.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new IllegalArgumentException("Failed get GPU information", e);
                }
            }
            is.close();
        }
        return stringArray;
    }

    private static void checkCall(int ret) {
        if (LIB == null) {
            throw new IllegalStateException("No cuda library is loaded.");
        }
        if (ret != 0) {
            throw new EngineException("CUDA API call failed: " + LIB.cudaGetErrorString(ret) + " (" + ret + ')');
        }
    }

    static {
        logging = true;
    }
}

