/**
 * FreeRDP: A Remote Desktop Protocol Implementation
 * common helper utilities
 *
 * Copyright 2024 Armin Novak <anovak@thincast.com>
 * Copyright 2024 Thincast Technologies GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#pragma once

#include <winpr/wtypes.h>
#include <winpr/json.h>

#include <freerdp/api.h>

#ifdef __cplusplus
extern "C"
{
#endif

	/** @brief Return the absolute path of a configuration file (the path of the configuration
	 * directory if \b filename is \b NULL)
	 *
	 *  @param system a boolean indicating the configuration base, \b TRUE for system configuration,
	 * \b FALSE for user configuration
	 *  @param filename an optional configuration file name to append.
	 *
	 *  @return The absolute path of the desired configuration or \b NULL in case of failure. Use \b
	 * free to clean up the allocated string.
	 *
	 *
	 *  @since version 3.9.0
	 */
	WINPR_ATTR_MALLOC(free, 1)
	WINPR_ATTR_NODISCARD
	FREERDP_API char* freerdp_GetConfigFilePath(BOOL system, const char* filename);

	/** @brief return a parsed JSON for a given config file name.
	 *
	 *  @param system a boolean indicating the configuration base, \b TRUE for system configuration,
	 * \b FALSE for user configuration
	 *  @param filename an optional configuration file name to append.
	 *
	 *  @return A parsed \b WINPR_JSON object or \b NULL in case of any failure.
	 *  @since version 3.16.0
	 */
	WINPR_ATTR_MALLOC(WINPR_JSON_Delete, 1)
	WINPR_ATTR_NODISCARD
	FREERDP_API WINPR_JSON* freerdp_GetJSONConfigFile(BOOL system, const char* filename);

	/** @brief set \b vendor and \b product information for an application
	 *
	 * This sets the application details for an application instance. These values determine where
	 * to look for configuration files and other vendor/product specific settings data.
	 * This function recursively also sets \ref freerdp_setApplicationDetails with a 'vendor' string
	 * of 'vendor/product', a 'product' string of WINPR_PRODUCT_STRING (build time constant) and a
	 * 'version' of -1. This limits the length of \b vendor + \b product to \b MAX_PATH or less..
	 *
	 * @note When calling this function, the compile time options \b
	 * FREERDP_USE_VENDOR_PRODUCT_CONFIG_DIR and \b WITH_FULL_CONFIG_PATH are ignored and the config
	 * path will always have the format 'vendor/product' or 'vendor/product1' (1 for the actual
	 * version set)
	 *
	 * @param vendor A vendor name to use. Must not be \b NULL. Must not contain forbidden
	 * filesystem symbols for any os. Must be less than \b MAX_PATH bytes.
	 * @param product A product name to use. Must not be \b NULL. Must not contain forbidden
	 * filesystem symbols for any os. Must be less than \b MAX_PATH bytes.
	 * @param version An optional versioning value to append to paths to settings. Use \b -1 to
	 * disable.
	 *
	 * @return \b TRUE if set successfully, \b FALSE in case of any error.
	 * @since version 3.23.0
	 */
	FREERDP_API WINPR_ATTR_NODISCARD BOOL freerdp_setApplicationDetails(const char* vendor,
	                                                                    const char* product,
	                                                                    SSIZE_T version);

	/** @brief Get the current \b vendor string of the application. Defaults to \ref
	 * FREERDP_VENDOR_STRING
	 *
	 * @return The current string set as \b vendor.
	 * @since version 3.23.0
	 */
	FREERDP_API WINPR_ATTR_NODISCARD const char* freerdp_getApplicationDetailsVendor(void);

	/** @brief Get the current \b product string of the application. Defaults to \ref
	 * FREERDP_PRODUCT_STRING
	 *
	 * @return The current string set as \b product.
	 * @since version 3.23.0
	 */
	FREERDP_API WINPR_ATTR_NODISCARD const char* freerdp_getApplicationDetailsProduct(void);

	/** @brief Get the current \b version of the application. Defaults to \ref FREERDP_API_VERSION
	 * if \b WITH_RESOURCE_VERSIONING is defined, otherwise \b -1
	 *
	 * @return The current number set as \b version
	 * @since version 3.23.0
	 */
	FREERDP_API WINPR_ATTR_NODISCARD SSIZE_T freerdp_getApplicationDetailsVersion(void);

	/** @brief Get the current details of the application.
	 *  Defaults to \b <Vendor>-<Product><Version> if \b WITH_RESOURCE_VERSIONING is defined,
	 *  \b <Vendor>-<Product> if <Vendor> does not equal <Product> and <Product> otherwise.
	 *
	 * @return The current application details as string.
	 * @since version 3.23.0
	 */
	FREERDP_API WINPR_ATTR_NODISCARD const char* freerdp_getApplicationDetailsString(void);

	/** @brief Get the current details of the application. Wide character version.
	 *  See \ref freerdp_getApplicationDetailsString for details.
	 *
	 * @return The current application details as string.
	 * @since version 3.23.0
	 */
	FREERDP_API WINPR_ATTR_NODISCARD const WCHAR* freerdp_getApplicationDetailsStringW(void);

#ifdef __cplusplus
}
#endif
