﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/docdb/DocDBRequest.h>
#include <aws/docdb/DocDB_EXPORTS.h>
#include <aws/docdb/model/Filter.h>

#include <utility>

namespace Aws {
namespace DocDB {
namespace Model {

/**
 */
class DescribeCertificatesRequest : public DocDBRequest {
 public:
  AWS_DOCDB_API DescribeCertificatesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeCertificates"; }

  AWS_DOCDB_API Aws::String SerializePayload() const override;

 protected:
  AWS_DOCDB_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The user-supplied certificate identifier. If this parameter is specified,
   * information for only the specified certificate is returned. If this parameter is
   * omitted, a list of up to <code>MaxRecords</code> certificates is returned. This
   * parameter is not case sensitive.</p> <p>Constraints</p> <ul> <li> <p>Must match
   * an existing <code>CertificateIdentifier</code>.</p> </li> </ul>
   */
  inline const Aws::String& GetCertificateIdentifier() const { return m_certificateIdentifier; }
  inline bool CertificateIdentifierHasBeenSet() const { return m_certificateIdentifierHasBeenSet; }
  template <typename CertificateIdentifierT = Aws::String>
  void SetCertificateIdentifier(CertificateIdentifierT&& value) {
    m_certificateIdentifierHasBeenSet = true;
    m_certificateIdentifier = std::forward<CertificateIdentifierT>(value);
  }
  template <typename CertificateIdentifierT = Aws::String>
  DescribeCertificatesRequest& WithCertificateIdentifier(CertificateIdentifierT&& value) {
    SetCertificateIdentifier(std::forward<CertificateIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>This parameter is not currently supported.</p>
   */
  inline const Aws::Vector<Filter>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Vector<Filter>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Vector<Filter>>
  DescribeCertificatesRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersT = Filter>
  DescribeCertificatesRequest& AddFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace_back(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of records to include in the response. If more records
   * exist than the specified <code>MaxRecords</code> value, a pagination token
   * called a marker is included in the response so that the remaining results can be
   * retrieved.</p> <p>Default: 100</p> <p>Constraints:</p> <ul> <li> <p>Minimum:
   * 20</p> </li> <li> <p>Maximum: 100</p> </li> </ul>
   */
  inline int GetMaxRecords() const { return m_maxRecords; }
  inline bool MaxRecordsHasBeenSet() const { return m_maxRecordsHasBeenSet; }
  inline void SetMaxRecords(int value) {
    m_maxRecordsHasBeenSet = true;
    m_maxRecords = value;
  }
  inline DescribeCertificatesRequest& WithMaxRecords(int value) {
    SetMaxRecords(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional pagination token provided by a previous
   * <code>DescribeCertificates</code> request. If this parameter is specified, the
   * response includes only records beyond the marker, up to the value specified by
   * <code>MaxRecords</code>.</p>
   */
  inline const Aws::String& GetMarker() const { return m_marker; }
  inline bool MarkerHasBeenSet() const { return m_markerHasBeenSet; }
  template <typename MarkerT = Aws::String>
  void SetMarker(MarkerT&& value) {
    m_markerHasBeenSet = true;
    m_marker = std::forward<MarkerT>(value);
  }
  template <typename MarkerT = Aws::String>
  DescribeCertificatesRequest& WithMarker(MarkerT&& value) {
    SetMarker(std::forward<MarkerT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_certificateIdentifier;

  Aws::Vector<Filter> m_filters;

  int m_maxRecords{0};

  Aws::String m_marker;
  bool m_certificateIdentifierHasBeenSet = false;
  bool m_filtersHasBeenSet = false;
  bool m_maxRecordsHasBeenSet = false;
  bool m_markerHasBeenSet = false;
};

}  // namespace Model
}  // namespace DocDB
}  // namespace Aws
