/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright(C) 2009,...,2018 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


#pragma once

/////////////////////// Stdlib includes
#include <vector>
#include <utility>
#include <algorithm>


/////////////////////// Qt includes
#include <QString>


/////////////////////// pappsomspp includes


/////////////////////// Local includes
#include "MsXpS/libXpertMassCore/jsclassregistrar.h"
#include "MsXpS/export-import-config.h"
#include "MsXpS/libXpertMassCore/globals.hpp"
#include "MsXpS/libXpertMassCore/IndexRange.hpp"

namespace MsXpS
{
namespace libXpertMassCore
{

  /*  BEGIN CLASS JS REFERENCE
   *  namespace: MsXpS::libXpertMassCore
   *  class name: IndexRangeCollection
   */

class DECLSPEC IndexRangeCollection: public QObject
{
  Q_OBJECT

  Q_PROPERTY(QString comment WRITE setComment READ getComment)

  friend class Polymer;
  friend class Oligomer;

  public:
  Q_INVOKABLE explicit IndexRangeCollection(QObject *parent = nullptr);
  Q_INVOKABLE explicit IndexRangeCollection(qsizetype index_start,
                                            qsizetype index_stop,
                                            QObject *parent = nullptr);
  Q_INVOKABLE explicit IndexRangeCollection(
    const QString &index_ranges_string,
    Enums::LocationType location_type = Enums::LocationType::INDEX,
    QObject *parent                   = nullptr);
  // Pseudo copy constructor
  Q_INVOKABLE explicit IndexRangeCollection(const IndexRangeCollection &other,
                                            QObject *parent = nullptr);

  virtual ~IndexRangeCollection();

  IndexRangeCollection &initialize(const IndexRangeCollection &other);
  IndexRangeCollection *clone(QObject *parent = nullptr);
  static IndexRangeCollection *clone(const IndexRangeCollection &other,
                                     QObject *parent = nullptr);


  void setComment(const QString &comment);
  QString getComment() const;

  Q_INVOKABLE const QList<IndexRange *> &getRangesCstRef() const;
  Q_INVOKABLE QList<IndexRange *> &getRangesRef();

  Q_INVOKABLE static QList<IndexRange *>
  parseIndexRanges(const QString &index_ranges_string,
                   Enums::LocationType location_type,
                   QObject *parent = nullptr);

  //////////////// OPERATORS /////////////////////
  IndexRangeCollection &operator=(const IndexRangeCollection &other) = delete;
  Q_INVOKABLE bool operator==(const IndexRangeCollection &other) const;
  Q_INVOKABLE bool operator!=(const IndexRangeCollection &other) const;

  //////////////// SEQUENCE RANGES and COORDINATES /////////////////////
  Q_INVOKABLE void setIndexRange(qsizetype start, qsizetype stop);
  Q_INVOKABLE void setIndexRange(const IndexRange &index_range);
  Q_INVOKABLE void setIndexRanges(const QList<IndexRange *> &index_ranges);
  Q_INVOKABLE void setIndexRanges(const IndexRangeCollection &index_ranges);
  Q_INVOKABLE qsizetype setIndexRanges(const QString &index_ranges_string,
                                       Enums::LocationType location_type);

  Q_INVOKABLE void appendIndexRange(qsizetype start, qsizetype stop);
  Q_INVOKABLE void appendIndexRange(const IndexRange &index_range);
  Q_INVOKABLE void appendIndexRanges(const QList<IndexRange *> &index_ranges);
  Q_INVOKABLE void appendIndexRanges(const IndexRangeCollection &index_ranges);

  //////////////// ACCESSING FUNCTIONS /////////////////////
  const IndexRange &getRangeCstRefAt(qsizetype index) const;
  IndexRange &getRangeRefAt(qsizetype index);
  IndexRange *getRangeAt(qsizetype index);

  const QList<IndexRange *>::const_iterator
  getRangeCstIteratorAt(qsizetype index) const;
  const QList<IndexRange *>::iterator getRangeIteratorAt(qsizetype index);

  Q_INVOKABLE qsizetype leftMostIndexRangeStart() const;
  Q_INVOKABLE QList<qsizetype> indicesOfLeftMostIndexRanges() const;
  Q_INVOKABLE bool isLeftMostIndexRange(const IndexRange &index_range) const;

  Q_INVOKABLE qsizetype rightMostIndexRangeStop() const;
  Q_INVOKABLE QList<qsizetype> indicesOfRightMostIndexRanges() const;
  Q_INVOKABLE bool isRightMostIndexRange(const IndexRange &index_range) const;

  Q_INVOKABLE IndexRange *mostInclusiveLeftRightIndexRange() const;

  Q_INVOKABLE bool encompassIndex(qsizetype index, bool globally = false) const;

  Q_INVOKABLE bool overlap() const;

  Q_INVOKABLE QString indicesAsText() const;
  Q_INVOKABLE QString positionsAsText() const;

  qsizetype size() const;

  void clear();

  static void registerJsConstructor(QJSEngine *engine);

  protected:
  QList<IndexRange *> m_ranges;
  QString m_comment;
};

  /*  END CLASS JS REFERENCE
   *  namespace: MsXpS::libXpertMassCore
   *  class name: IndexRangeCollection
   */


} // namespace libXpertMassCore
MSXPS_REGISTER_JS_CLASS(MsXpS::libXpertMassCore, IndexRangeCollection)

} // namespace MsXpS
