package cu

// This file provides access to CUDA driver error statuses (type CUresult).

//#include <cuda.h>
import "C"
import (
	"fmt"
)

// CUDA error status.
// CUDA error statuses are not returned by functions but checked and passed to
// panic() when not successful. If desired, they can be caught by
// recover().
type Result int

// Message string for the error
func (err Result) String() string {
	str, ok := errorString[err]
	if !ok {
		return "Unknown CUresult: " + fmt.Sprint(int(err))
	}
	return str
}

const (
	SUCCESS                              Result = C.CUDA_SUCCESS
	ERROR_INVALID_VALUE                  Result = C.CUDA_ERROR_INVALID_VALUE
	ERROR_OUT_OF_MEMORY                  Result = C.CUDA_ERROR_OUT_OF_MEMORY
	ERROR_NOT_INITIALIZED                Result = C.CUDA_ERROR_NOT_INITIALIZED
	ERROR_DEINITIALIZED                  Result = C.CUDA_ERROR_DEINITIALIZED
	ERROR_PROFILER_DISABLED              Result = C.CUDA_ERROR_PROFILER_DISABLED
	ERROR_PROFILER_NOT_INITIALIZED       Result = C.CUDA_ERROR_PROFILER_NOT_INITIALIZED
	ERROR_PROFILER_ALREADY_STARTED       Result = C.CUDA_ERROR_PROFILER_ALREADY_STARTED
	ERROR_PROFILER_ALREADY_STOPPED       Result = C.CUDA_ERROR_PROFILER_ALREADY_STOPPED
	ERROR_NO_DEVICE                      Result = C.CUDA_ERROR_NO_DEVICE
	ERROR_INVALID_DEVICE                 Result = C.CUDA_ERROR_INVALID_DEVICE
	ERROR_INVALID_IMAGE                  Result = C.CUDA_ERROR_INVALID_IMAGE
	ERROR_INVALID_CONTEXT                Result = C.CUDA_ERROR_INVALID_CONTEXT
	ERROR_CONTEXT_ALREADY_CURRENT        Result = C.CUDA_ERROR_CONTEXT_ALREADY_CURRENT
	ERROR_MAP_FAILED                     Result = C.CUDA_ERROR_MAP_FAILED
	ERROR_UNMAP_FAILED                   Result = C.CUDA_ERROR_UNMAP_FAILED
	ERROR_ARRAY_IS_MAPPED                Result = C.CUDA_ERROR_ARRAY_IS_MAPPED
	ERROR_ALREADY_MAPPED                 Result = C.CUDA_ERROR_ALREADY_MAPPED
	ERROR_NO_BINARY_FOR_GPU              Result = C.CUDA_ERROR_NO_BINARY_FOR_GPU
	ERROR_ALREADY_ACQUIRED               Result = C.CUDA_ERROR_ALREADY_ACQUIRED
	ERROR_NOT_MAPPED                     Result = C.CUDA_ERROR_NOT_MAPPED
	ERROR_NOT_MAPPED_AS_ARRAY            Result = C.CUDA_ERROR_NOT_MAPPED_AS_ARRAY
	ERROR_NOT_MAPPED_AS_POINTER          Result = C.CUDA_ERROR_NOT_MAPPED_AS_POINTER
	ERROR_ECC_UNCORRECTABLE              Result = C.CUDA_ERROR_ECC_UNCORRECTABLE
	ERROR_UNSUPPORTED_LIMIT              Result = C.CUDA_ERROR_UNSUPPORTED_LIMIT
	ERROR_CONTEXT_ALREADY_IN_USE         Result = C.CUDA_ERROR_CONTEXT_ALREADY_IN_USE
	ERROR_INVALID_SOURCE                 Result = C.CUDA_ERROR_INVALID_SOURCE
	ERROR_FILE_NOT_FOUND                 Result = C.CUDA_ERROR_FILE_NOT_FOUND
	ERROR_SHARED_OBJECT_SYMBOL_NOT_FOUND Result = C.CUDA_ERROR_SHARED_OBJECT_SYMBOL_NOT_FOUND
	ERROR_SHARED_OBJECT_INIT_FAILED      Result = C.CUDA_ERROR_SHARED_OBJECT_INIT_FAILED
	ERROR_OPERATING_SYSTEM               Result = C.CUDA_ERROR_OPERATING_SYSTEM
	ERROR_INVALID_HANDLE                 Result = C.CUDA_ERROR_INVALID_HANDLE
	ERROR_NOT_FOUND                      Result = C.CUDA_ERROR_NOT_FOUND
	ERROR_NOT_READY                      Result = C.CUDA_ERROR_NOT_READY
	ERROR_LAUNCH_FAILED                  Result = C.CUDA_ERROR_LAUNCH_FAILED
	ERROR_LAUNCH_OUT_OF_RESOURCES        Result = C.CUDA_ERROR_LAUNCH_OUT_OF_RESOURCES
	ERROR_LAUNCH_TIMEOUT                 Result = C.CUDA_ERROR_LAUNCH_TIMEOUT
	ERROR_LAUNCH_INCOMPATIBLE_TEXTURING  Result = C.CUDA_ERROR_LAUNCH_INCOMPATIBLE_TEXTURING
	ERROR_PEER_ACCESS_ALREADY_ENABLED    Result = C.CUDA_ERROR_PEER_ACCESS_ALREADY_ENABLED
	ERROR_PEER_ACCESS_NOT_ENABLED        Result = C.CUDA_ERROR_PEER_ACCESS_NOT_ENABLED
	ERROR_PRIMARY_CONTEXT_ACTIVE         Result = C.CUDA_ERROR_PRIMARY_CONTEXT_ACTIVE
	ERROR_CONTEXT_IS_DESTROYED           Result = C.CUDA_ERROR_CONTEXT_IS_DESTROYED
	ERROR_ASSERT                         Result = C.CUDA_ERROR_ASSERT
	ERROR_TOO_MANY_PEERS                 Result = C.CUDA_ERROR_TOO_MANY_PEERS
	ERROR_HOST_MEMORY_ALREADY_REGISTERED Result = C.CUDA_ERROR_HOST_MEMORY_ALREADY_REGISTERED
	ERROR_HOST_MEMORY_NOT_REGISTERED     Result = C.CUDA_ERROR_HOST_MEMORY_NOT_REGISTERED
	ERROR_HARDWARE_STACK_ERROR           Result = 714 //C.CUDA_ERROR_HARDWARE_STACK_ERROR
	ERROR_ILLEGAL_INSTRUCTION            Result = 715 //C.CUDA_ERROR_ILLEGAL_INSTRUCTION
	ERROR_MISALIGNED_ADDRESS             Result = 716 //C.CUDA_ERROR_MISALIGNED_ADDRESS
	ERROR_INVALID_ADDRESS_SPACE          Result = 717 //C.CUDA_ERROR_INVALID_ADDRESS_SPACE
	ERROR_INVALID_PC                     Result = 718 //C.CUDA_ERROR_INVALID_PC
	ERROR_NOT_PERMITTED                  Result = 800 //C.CUDA_ERROR_NOT_PERMITTED
	ERROR_NOT_SUPPORTED                  Result = 801 //C.CUDA_ERROR_NOT_SUPPORTED
	ERROR_UNKNOWN                        Result = C.CUDA_ERROR_UNKNOWN
)

// Map with error strings for Result error numbers
var errorString map[Result]string = map[Result]string{
	SUCCESS:                              "CUDA_SUCCESS",
	ERROR_INVALID_VALUE:                  "CUDA_ERROR_INVALID_VALUE",
	ERROR_OUT_OF_MEMORY:                  "CUDA_ERROR_OUT_OF_MEMORY",
	ERROR_NOT_INITIALIZED:                "CUDA_ERROR_NOT_INITIALIZED",
	ERROR_DEINITIALIZED:                  "CUDA_ERROR_DEINITIALIZED",
	ERROR_PROFILER_DISABLED:              "CUDA_ERROR_PROFILER_DISABLED",
	ERROR_PROFILER_NOT_INITIALIZED:       "CUDA_ERROR_PROFILER_NOT_INITIALIZED",
	ERROR_PROFILER_ALREADY_STARTED:       "CUDA_ERROR_PROFILER_ALREADY_STARTED",
	ERROR_PROFILER_ALREADY_STOPPED:       "CUDA_ERROR_PROFILER_ALREADY_STOPPED",
	ERROR_NO_DEVICE:                      "CUDA_ERROR_NO_DEVICE",
	ERROR_INVALID_DEVICE:                 "CUDA_ERROR_INVALID_DEVICE",
	ERROR_INVALID_IMAGE:                  "CUDA_ERROR_INVALID_IMAGE",
	ERROR_INVALID_CONTEXT:                "CUDA_ERROR_INVALID_CONTEXT",
	ERROR_CONTEXT_ALREADY_CURRENT:        "CUDA_ERROR_CONTEXT_ALREADY_CURRENT",
	ERROR_MAP_FAILED:                     "CUDA_ERROR_MAP_FAILED",
	ERROR_UNMAP_FAILED:                   "CUDA_ERROR_UNMAP_FAILED",
	ERROR_ARRAY_IS_MAPPED:                "CUDA_ERROR_ARRAY_IS_MAPPED",
	ERROR_ALREADY_MAPPED:                 "CUDA_ERROR_ALREADY_MAPPED",
	ERROR_NO_BINARY_FOR_GPU:              "CUDA_ERROR_NO_BINARY_FOR_GPU",
	ERROR_ALREADY_ACQUIRED:               "CUDA_ERROR_ALREADY_ACQUIRED",
	ERROR_NOT_MAPPED:                     "CUDA_ERROR_NOT_MAPPED",
	ERROR_NOT_MAPPED_AS_ARRAY:            "CUDA_ERROR_NOT_MAPPED_AS_ARRAY",
	ERROR_NOT_MAPPED_AS_POINTER:          "CUDA_ERROR_NOT_MAPPED_AS_POINTER",
	ERROR_ECC_UNCORRECTABLE:              "CUDA_ERROR_ECC_UNCORRECTABLE",
	ERROR_UNSUPPORTED_LIMIT:              "CUDA_ERROR_UNSUPPORTED_LIMIT",
	ERROR_CONTEXT_ALREADY_IN_USE:         "CUDA_ERROR_CONTEXT_ALREADY_IN_USE",
	ERROR_INVALID_SOURCE:                 "CUDA_ERROR_INVALID_SOURCE",
	ERROR_FILE_NOT_FOUND:                 "CUDA_ERROR_FILE_NOT_FOUND",
	ERROR_SHARED_OBJECT_SYMBOL_NOT_FOUND: "CUDA_ERROR_SHARED_OBJECT_SYMBOL_NOT_FOUND",
	ERROR_SHARED_OBJECT_INIT_FAILED:      "CUDA_ERROR_SHARED_OBJECT_INIT_FAILED",
	ERROR_OPERATING_SYSTEM:               "CUDA_ERROR_OPERATING_SYSTEM",
	ERROR_INVALID_HANDLE:                 "CUDA_ERROR_INVALID_HANDLE",
	ERROR_NOT_FOUND:                      "CUDA_ERROR_NOT_FOUND",
	ERROR_NOT_READY:                      "CUDA_ERROR_NOT_READY",
	ERROR_LAUNCH_OUT_OF_RESOURCES:        "CUDA_ERROR_LAUNCH_OUT_OF_RESOURCES",
	ERROR_LAUNCH_TIMEOUT:                 "CUDA_ERROR_LAUNCH_TIMEOUT",
	ERROR_LAUNCH_INCOMPATIBLE_TEXTURING:  "CUDA_ERROR_LAUNCH_INCOMPATIBLE_TEXTURING",
	ERROR_PEER_ACCESS_ALREADY_ENABLED:    "CUDA_ERROR_PEER_ACCESS_ALREADY_ENABLED",
	ERROR_PEER_ACCESS_NOT_ENABLED:        "CUDA_ERROR_PEER_ACCESS_NOT_ENABLED",
	ERROR_PRIMARY_CONTEXT_ACTIVE:         "CUDA_ERROR_PRIMARY_CONTEXT_ACTIVE",
	ERROR_CONTEXT_IS_DESTROYED:           "CUDA_ERROR_CONTEXT_IS_DESTROYED",
	ERROR_ASSERT:                         "CUDA_ERROR_ASSERT",
	ERROR_TOO_MANY_PEERS:                 "CUDA_ERROR_TOO_MANY_PEERS",
	ERROR_HOST_MEMORY_ALREADY_REGISTERED: "CUDA_ERROR_HOST_MEMORY_ALREADY_REGISTERED",
	ERROR_HOST_MEMORY_NOT_REGISTERED:     "CUDA_ERROR_HOST_MEMORY_NOT_REGISTERED",
	ERROR_HARDWARE_STACK_ERROR:           "CUDA_ERROR_HARDWARE_STACK_ERROR",
	ERROR_ILLEGAL_INSTRUCTION:            "CUDA_ERROR_ILLEGAL_INSTRUCTION",
	ERROR_MISALIGNED_ADDRESS:             "CUDA_ERROR_MISALIGNED_ADDRESS",
	ERROR_INVALID_ADDRESS_SPACE:          "CUDA_ERROR_INVALID_ADDRESS_SPACE",
	ERROR_INVALID_PC:                     "CUDA_ERROR_INVALID_PC",
	ERROR_LAUNCH_FAILED:                  "CUDA_ERROR_LAUNCH_FAILED",
	ERROR_NOT_PERMITTED:                  "CUDA_ERROR_NOT_PERMITTED",
	ERROR_NOT_SUPPORTED:                  "CUDA_ERROR_NOT_SUPPORTED",
	ERROR_UNKNOWN:                        "CUDA_ERROR_UNKNOWN"}
