# main.py
#
# Copyright 2024 Hari Rana (TheEvilSkeleton)
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.
#
# SPDX-License-Identifier: GPL-3.0-or-later

import logging
import sys
from collections.abc import Callable
from gettext import gettext as _
from typing import Any

import gi

from refine import APP_ID, MAJOR, MINOR, PROFILE

gi.require_version("Gtk", "4.0")
gi.require_version("Adw", "1")
gi.require_version("Pango", "1.0")
gi.require_version("XdpGtk4", "1.0")

# ruff: noqa: E402
from gi.repository import Adw, Gio

from refine import settings
from refine.typing import SettingsSource
from refine.window import Window


class Application(Adw.Application):
    """The main application singleton class."""

    settings: dict[str, SettingsSource] = {}

    def __init__(self) -> None:
        super().__init__(
            application_id=APP_ID,
            flags=Gio.ApplicationFlags.DEFAULT_FLAGS,
            resource_base_path="/page/tesk/Refine",
        )
        self.create_action("quit", lambda *_: self.quit(), ["<primary>q"])
        self.create_action("about", self.on_about_action)

        self.sources: list[Gio.SettingsSchemaSource] = (
            settings.create_schema_sources_list()
        )

    def do_activate(self) -> None:
        """
        Call when the application is activated.

        We raise the application's main window, creating it if necessary.
        """
        win = self.props.active_window
        if not win:
            win = Window(application=self)

        win.present()

        if PROFILE == "development":
            logging.getLogger().setLevel(logging.DEBUG)
            win.add_css_class("devel")

    def on_about_action(self, *_args: Any) -> None:
        """Callback for the app.about action."""
        about = Adw.AboutDialog.new_from_appdata(
            "/page/tesk/Refine/appdata", f"{MAJOR}.{MINOR}.0"
        )

        # Translators: Replace "translator-credits" with your name/username, and optionally an email or URL.
        about.set_translator_credits(_("translator-credits"))
        about.set_copyright(_("Copyright © 2024 Hari Rana (TheEvilSkeleton)"))

        about.add_other_app(  # type: ignore [attr-defined]
            "com.usebottles.bottles",
            _("Bottles"),
            _("Run Windows software"),
        )
        about.add_other_app(  # type: ignore [attr-defined]
            "io.gitlab.theevilskeleton.Upscaler",
            _("Upscaler"),
            _("Upscale and enhance images"),
        )

        about.set_developers(
            (
                f"{about.get_developer_name()} https://tesk.page",
                "Evelyn Serra (itsEve) https://itseve.xyz",
                "Mitchell Winkelman (Heliguy) https://heliguy.neocities.org",
            )
        )
        about.set_artists(("Dominik Baran https://gitlab.gnome.org/toad",))
        about.set_debug_info(settings.get_debug_info())
        about.present(self.props.active_window)

    def create_action(
        self,
        name: str,
        callback: Callable[[Any], None],
        shortcuts: None | list[str] = None,
    ) -> None:
        """Add an application action."""
        action = Gio.SimpleAction.new(name, None)
        action.connect("activate", callback)
        self.add_action(action)
        if shortcuts:
            self.set_accels_for_action(f"app.{name}", shortcuts)


def main(version: str) -> int:
    """The application's entry point."""
    app = Application()
    return app.run(sys.argv)
