#lang scribble/doc
@(require "common.rkt")

@defclass/title[font% object% ()]{

A @defterm{font} is an object which determines the appearance of text,
 primarily when drawing text to a device context. A font is determined
 by eight properties:

@itemize[

 @item{size --- The size of the text, either in points (the default)
                or logical drawing units, depending on the
                ``size-in-pixels?'' property (see below).}

 @item{family --- A platform- and device-independent font
                  designation. The families are:

 @itemize[
 @item{@indexed-racket['default]}
 @item{@indexed-racket['decorative]}
 @item{@indexed-racket['roman]}
 @item{@indexed-racket['script]}
 @item{@indexed-racket['swiss]}
 @item{@indexed-racket['modern] (fixed width)}
 @item{@indexed-racket['symbol] (Greek letters and more)}
 @item{@indexed-racket['system] (similar to the font to draw control labels,
                                 but see @racket[normal-control-font])}
 ]

 @margin-note{The terminology ``family'' and ``face'' is mangled
              relative to its usual meaning. A @racket[font%] ``face''
              is really used more like a font family in the usual
              terminology or more generally as a face-description
              string that is combined with other @racket[font%]
              attributes to arrive at a face. A @racket[font%]
              ``family'' is a kind of abstract font family that is
              mapped to a particular font family on a given
              platform.}}

 @item{face --- A string face name, such as @racket["Courier"]. The format
                and meaning of a face name is platform- and
                device-specific. If a font's face name is @racket[#f],
                then the font's appearance depends only on the
                family. If a face is provided but no mapping is
                available for the face name (for a specific platform
                or device), then the face name is ignored and the
                family is used. See @racket[font-name-directory<%>]
                for information about how face names are mapped for
                drawing text.}

@item{style --- The slant style of the font, one of:
 @itemize[
 @item{@indexed-racket['normal]}
 @item{@indexed-racket['slant] (a.k.a ``oblique'')}
 @item{@indexed-racket['italic]}
 ]}

@item{@deftech[#:key "font weight"]{weight} ---
 The weight of the font, one of:
 @itemize[
 @item{@racket[(integer-in 100 1000)]}
 @item{@indexed-racket['thin] (equivalent to @racket[100])}
 @item{@indexed-racket['ultralight] (equivalent to @racket[200])}
 @item{@indexed-racket['light] (equivalent to @racket[300])}
 @item{@indexed-racket['semilight] (equivalent to @racket[350])}
 @item{@indexed-racket['book] (equivalent to @racket[380])}
 @item{@indexed-racket['normal] (equivalent to @racket[400])}
 @item{@indexed-racket['medium] (equivalent to @racket[500])}
 @item{@indexed-racket['semibold] (equivalent to @racket[600])}
 @item{@indexed-racket['bold] (equivalent to @racket[700])}
 @item{@indexed-racket['ultrabold] (equivalent to @racket[800])}
 @item{@indexed-racket['heavy] (equivalent to @racket[900])}
 @item{@indexed-racket['ultraheavy] (equivalent to @racket[1000])}
 ]

 @history[#:changed "1.14" @elem{Changed to allow integer values and the symbols @racket['thin],
                                 @racket['ultralight], @racket['semilight], @racket['book],
                                 @racket['medium], @racket['semibold], @racket['ultrabold],
                                 @racket['heavy], and @racket['ultraheavy].}]
 }

@item{underline? --- @racket[#t] for underlined, @racket[#f] for plain.}

@item{smoothing --- Amount of anti-alias smoothing, one of:
 @itemize[
 @item{@indexed-racket['default] (platform-specific, sometimes user-configurable)}
 @item{@indexed-racket['partly-smoothed] (gray anti-aliasing)}
 @item{@indexed-racket['smoothed] (sub-pixel anti-aliasing)}
 @item{@indexed-racket['unsmoothed]}
 ]}

@item{size-in-pixels? --- @racket[#t] if the size of the font
 is in logical drawing units (i.e., pixels for an unscaled screen or
 bitmap drawing context), @racket[#f] if the size of the font is in
 @deftech[#:key "font point"]{points}, where a point is equal to 1
 pixel on Mac OS and @racket[(/ 96 72)] pixels on Windows and Unix}

@item{hinting --- Whether font metrics should be rounded to integers:
 @itemize[
   @item{@indexed-racket['aligned] (the default) --- rounds to integers
    to improve the consistency of letter spacing for pixel-based
    targets, but at the expense of making metrics unscalable}
   @item{@indexed-racket['unaligned] --- disables rounding}
 ]}

@item{@deftech[#:key "OpenType feature settings"]{feature settings} ---
        A hash of @hyperlink["https://practicaltypography.com/opentype-features.html"]{
        OpenType feature} settings to enable or disable optional typographic
        features of OpenType fonts. Each entry in the hash maps a four-letter
        OpenType feature tag to its desired value. For boolean OpenType features,
        a value of @racket[0] means “disabled” and a value of @racket[1] means
        “enabled”; for other features, the meaning of the value varies (and may
        even depend on the font itself).

        @history[#:added "1.19"]}

]

To avoid creating multiple fonts with the same characteristics, use
 the global @racket[font-list%] object @indexed-racket[the-font-list].

See also
@racket[font-name-directory<%>].

@history[#:changed "1.2" @elem{Defined ``points'' as  @racket[(/ 96 72)] pixels on Windows,
                               independent of the screen resolution.}
         #:changed "1.14" @elem{Changed ``weight'' to allow integer values and the symbols
                                @racket['thin], @racket['ultralight], @racket['semilight],
                                @racket['book], @racket['medium], @racket['semibold],
                                @racket['ultrabold], @racket['heavy], and @racket['ultraheavy].}
         #:changed "1.19" @elem{Added the ``feature settings'' property to control OpenType features.}]


@defconstructor*/make[(()
                       ([size (real-in 0.0 1024.0)]
                        [family font-family/c]
                        [style font-style/c 'normal]
                        [weight font-weight/c 'normal]
                        [underline? any/c #f]
                        [smoothing font-smoothing/c 'default]
                        [size-in-pixels? any/c #f]
                        [hinting font-hinting/c 'aligned]
                        [feature-settings font-feature-settings/c (hash)])
                       ([size (real-in 0.0 1024.0)]
                        [face string?]
                        [family font-family/c]
                        [style font-style/c 'normal]
                        [weight font-weight/c 'normal]
                        [underline? any/c #f]
                        [smoothing font-smoothing/c 'default]
                        [size-in-pixels? any/c #f]
                        [hinting font-hinting/c 'aligned]
                        [feature-settings font-feature-settings/c (hash)]))]{

When no arguments are provided, creates an instance of the default
 font. If no face name is provided, the font is created without a face
 name.

See @racket[font%] for information about @racket[family],
 @racket[style], @racket[weight], @racket[smoothing],
 @racket[size-in-pixels?], @racket[hinting], and @racket[feature-settings].
 @racket[font-name-directory<%>].

See also @racket[make-font].

@history[#:changed "1.4" @elem{Changed @racket[size] to allow non-integer and zero values.}
         #:changed "1.14" @elem{Changed @racket[weight] to allow integer values and the symbols
                                @racket['thin], @racket['ultralight], @racket['semilight],
                                @racket['book], @racket['medium], @racket['semibold],
                                @racket['ultrabold], @racket['heavy], and @racket['ultraheavy].}
         #:changed "1.19" @elem{Added the optional @racket[feature-settings] argument.}]}


@defmethod[(get-face)
           (or/c string? #f)]{

Gets the font's face name, or @racket[#f] if none is specified.

}

@defmethod[(get-family) font-family/c]{

Gets the font's family. See @racket[font%] for information about
families.

}

@defmethod[(get-feature-settings) font-feature-settings/c]{

Gets the font's @tech{OpenType feature settings}.

}

@defmethod[(get-font-id)
           exact-integer?]{

Gets the font's ID, for use with a
@racket[font-name-directory<%>]. The ID is determined by the font's
face and family specifications, only.

}

@defmethod[(get-hinting) font-hinting/c]{

Gets the font's hinting. See @racket[font%] for information about
hinting.

}

@defmethod[(get-point-size)
           (integer-in 1 1024)]{

Gets the font's size rounded to the nearest non-zero integer. Despite
 the method's name, the result is in either logical units or
 @tech[#:key "font point"]{points}, depending on the result of
 @method[font% get-size-in-pixels].

See @method[font% get-size], instead. The @method[font% get-point-size]
 method is provided for backward compatibility.}


@defmethod[(get-size [in-pixels? any/c (send @#,this-obj[] @#,method[font% get-size-in-pixels])])
           (real-in 0.0 1024.0)]{

Gets the font's size (roughly the height). If @racket[in-pixels?] is
 @racket[#f], the size is in @tech[#:key "font point"]{points},
 otherwise it is in logical units.

Due to space included in a font by a font designer, a font tends to
 generate text that is slightly taller than the nominal size.

@history[#:added "1.4"
         #:changed "1.19" @elem{Added the @racket[in-pixels?] argument.}]}


@defmethod[(get-size-in-pixels)
           boolean?]{

Returns @racket[#t] if @method[font% get-size] defaults to reporting
 the size in logical drawing units, @racket[#f] if it defaults to
 reporting @tech[#:key "font point"]{points}.

}

@defmethod[(get-smoothing) font-smoothing/c]{

Gets the font's anti-alias smoothing mode. See @racket[font%] for
 information about smoothing.

}

@defmethod[(get-style) font-style/c]{

Gets the font's slant style. See @racket[font%] for information about
 styles.

}

@defmethod[(get-underlined)
           boolean?]{

Returns @racket[#t] if the font is underlined or @racket[#f]
otherwise.

}

@defmethod[(get-weight) font-weight/c]{

Gets the font's weight. See @racket[font%] for information about
 weights.

}

@defmethod[(screen-glyph-exists? [c char?]
                                 [for-label? any/c #f])
           boolean?]{

Returns @racket[#t] if the given character has a corresponding glyph
 when drawing to the screen or a bitmap, @racket[#f] otherwise.

If the second argument is true, the result indicates whether the glyph
 is available for control labels. Otherwise, it indicates whether the
 glyph is available for @racket[dc<%>] drawing.

For @racket[dc<%>] drawing, due to automatic font substitution when
 drawing or measuring text, the result of this method does not depend
 on this font's attributes (size, face, etc.). The font's attributes
 merely provide a hint for the glyph search.

See also @method[dc<%> glyph-exists?] .

}}
