﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/scheduler/Scheduler_EXPORTS.h>
#include <aws/scheduler/model/ActionAfterCompletion.h>
#include <aws/scheduler/model/FlexibleTimeWindow.h>
#include <aws/scheduler/model/ScheduleState.h>
#include <aws/scheduler/model/Target.h>

#include <utility>

namespace Aws {
template <typename RESULT_TYPE>
class AmazonWebServiceResult;

namespace Utils {
namespace Json {
class JsonValue;
}  // namespace Json
}  // namespace Utils
namespace Scheduler {
namespace Model {
class GetScheduleResult {
 public:
  AWS_SCHEDULER_API GetScheduleResult() = default;
  AWS_SCHEDULER_API GetScheduleResult(const Aws::AmazonWebServiceResult<Aws::Utils::Json::JsonValue>& result);
  AWS_SCHEDULER_API GetScheduleResult& operator=(const Aws::AmazonWebServiceResult<Aws::Utils::Json::JsonValue>& result);

  ///@{
  /**
   * <p>Indicates the action that EventBridge Scheduler applies to the schedule after
   * the schedule completes invoking the target.</p>
   */
  inline ActionAfterCompletion GetActionAfterCompletion() const { return m_actionAfterCompletion; }
  inline void SetActionAfterCompletion(ActionAfterCompletion value) {
    m_actionAfterCompletionHasBeenSet = true;
    m_actionAfterCompletion = value;
  }
  inline GetScheduleResult& WithActionAfterCompletion(ActionAfterCompletion value) {
    SetActionAfterCompletion(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the schedule.</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  GetScheduleResult& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time at which the schedule was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreationDate() const { return m_creationDate; }
  template <typename CreationDateT = Aws::Utils::DateTime>
  void SetCreationDate(CreationDateT&& value) {
    m_creationDateHasBeenSet = true;
    m_creationDate = std::forward<CreationDateT>(value);
  }
  template <typename CreationDateT = Aws::Utils::DateTime>
  GetScheduleResult& WithCreationDate(CreationDateT&& value) {
    SetCreationDate(std::forward<CreationDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the schedule.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  GetScheduleResult& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date, in UTC, before which the schedule can invoke its target. Depending
   * on the schedule's recurrence expression, invocations might stop on, or before,
   * the <code>EndDate</code> you specify. EventBridge Scheduler ignores
   * <code>EndDate</code> for one-time schedules.</p>
   */
  inline const Aws::Utils::DateTime& GetEndDate() const { return m_endDate; }
  template <typename EndDateT = Aws::Utils::DateTime>
  void SetEndDate(EndDateT&& value) {
    m_endDateHasBeenSet = true;
    m_endDate = std::forward<EndDateT>(value);
  }
  template <typename EndDateT = Aws::Utils::DateTime>
  GetScheduleResult& WithEndDate(EndDateT&& value) {
    SetEndDate(std::forward<EndDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Allows you to configure a time window during which EventBridge Scheduler
   * invokes the schedule.</p>
   */
  inline const FlexibleTimeWindow& GetFlexibleTimeWindow() const { return m_flexibleTimeWindow; }
  template <typename FlexibleTimeWindowT = FlexibleTimeWindow>
  void SetFlexibleTimeWindow(FlexibleTimeWindowT&& value) {
    m_flexibleTimeWindowHasBeenSet = true;
    m_flexibleTimeWindow = std::forward<FlexibleTimeWindowT>(value);
  }
  template <typename FlexibleTimeWindowT = FlexibleTimeWindow>
  GetScheduleResult& WithFlexibleTimeWindow(FlexibleTimeWindowT&& value) {
    SetFlexibleTimeWindow(std::forward<FlexibleTimeWindowT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the schedule group associated with this schedule.</p>
   */
  inline const Aws::String& GetGroupName() const { return m_groupName; }
  template <typename GroupNameT = Aws::String>
  void SetGroupName(GroupNameT&& value) {
    m_groupNameHasBeenSet = true;
    m_groupName = std::forward<GroupNameT>(value);
  }
  template <typename GroupNameT = Aws::String>
  GetScheduleResult& WithGroupName(GroupNameT&& value) {
    SetGroupName(std::forward<GroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN for a customer managed KMS Key that is be used to encrypt and decrypt
   * your data.</p>
   */
  inline const Aws::String& GetKmsKeyArn() const { return m_kmsKeyArn; }
  template <typename KmsKeyArnT = Aws::String>
  void SetKmsKeyArn(KmsKeyArnT&& value) {
    m_kmsKeyArnHasBeenSet = true;
    m_kmsKeyArn = std::forward<KmsKeyArnT>(value);
  }
  template <typename KmsKeyArnT = Aws::String>
  GetScheduleResult& WithKmsKeyArn(KmsKeyArnT&& value) {
    SetKmsKeyArn(std::forward<KmsKeyArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time at which the schedule was last modified.</p>
   */
  inline const Aws::Utils::DateTime& GetLastModificationDate() const { return m_lastModificationDate; }
  template <typename LastModificationDateT = Aws::Utils::DateTime>
  void SetLastModificationDate(LastModificationDateT&& value) {
    m_lastModificationDateHasBeenSet = true;
    m_lastModificationDate = std::forward<LastModificationDateT>(value);
  }
  template <typename LastModificationDateT = Aws::Utils::DateTime>
  GetScheduleResult& WithLastModificationDate(LastModificationDateT&& value) {
    SetLastModificationDate(std::forward<LastModificationDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the schedule.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  GetScheduleResult& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The expression that defines when the schedule runs. The following formats
   * are supported. </p> <ul> <li> <p> <code>at</code> expression -
   * <code>at(yyyy-mm-ddThh:mm:ss)</code> </p> </li> <li> <p> <code>rate</code>
   * expression - <code>rate(value unit)</code> </p> </li> <li> <p> <code>cron</code>
   * expression - <code>cron(fields)</code> </p> </li> </ul> <p> You can use
   * <code>at</code> expressions to create one-time schedules that invoke a target
   * once, at the time and in the time zone, that you specify. You can use
   * <code>rate</code> and <code>cron</code> expressions to create recurring
   * schedules. Rate-based schedules are useful when you want to invoke a target at
   * regular intervals, such as every 15 minutes or every five days. Cron-based
   * schedules are useful when you want to invoke a target periodically at a specific
   * time, such as at 8:00 am (UTC+0) every 1st day of the month. </p> <p> A
   * <code>cron</code> expression consists of six fields separated by white spaces:
   * <code>(minutes hours day_of_month month day_of_week year)</code>. </p> <p> A
   * <code>rate</code> expression consists of a <i>value</i> as a positive integer,
   * and a <i>unit</i> with the following options: <code>minute</code> |
   * <code>minutes</code> | <code>hour</code> | <code>hours</code> | <code>day</code>
   * | <code>days</code> </p> <p> For more information and examples, see <a
   * href="https://docs.aws.amazon.com/scheduler/latest/UserGuide/schedule-types.html">Schedule
   * types on EventBridge Scheduler</a> in the <i>EventBridge Scheduler User
   * Guide</i>. </p>
   */
  inline const Aws::String& GetScheduleExpression() const { return m_scheduleExpression; }
  template <typename ScheduleExpressionT = Aws::String>
  void SetScheduleExpression(ScheduleExpressionT&& value) {
    m_scheduleExpressionHasBeenSet = true;
    m_scheduleExpression = std::forward<ScheduleExpressionT>(value);
  }
  template <typename ScheduleExpressionT = Aws::String>
  GetScheduleResult& WithScheduleExpression(ScheduleExpressionT&& value) {
    SetScheduleExpression(std::forward<ScheduleExpressionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The timezone in which the scheduling expression is evaluated.</p>
   */
  inline const Aws::String& GetScheduleExpressionTimezone() const { return m_scheduleExpressionTimezone; }
  template <typename ScheduleExpressionTimezoneT = Aws::String>
  void SetScheduleExpressionTimezone(ScheduleExpressionTimezoneT&& value) {
    m_scheduleExpressionTimezoneHasBeenSet = true;
    m_scheduleExpressionTimezone = std::forward<ScheduleExpressionTimezoneT>(value);
  }
  template <typename ScheduleExpressionTimezoneT = Aws::String>
  GetScheduleResult& WithScheduleExpressionTimezone(ScheduleExpressionTimezoneT&& value) {
    SetScheduleExpressionTimezone(std::forward<ScheduleExpressionTimezoneT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date, in UTC, after which the schedule can begin invoking its target.
   * Depending on the schedule's recurrence expression, invocations might occur on,
   * or after, the <code>StartDate</code> you specify. EventBridge Scheduler ignores
   * <code>StartDate</code> for one-time schedules.</p>
   */
  inline const Aws::Utils::DateTime& GetStartDate() const { return m_startDate; }
  template <typename StartDateT = Aws::Utils::DateTime>
  void SetStartDate(StartDateT&& value) {
    m_startDateHasBeenSet = true;
    m_startDate = std::forward<StartDateT>(value);
  }
  template <typename StartDateT = Aws::Utils::DateTime>
  GetScheduleResult& WithStartDate(StartDateT&& value) {
    SetStartDate(std::forward<StartDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether the schedule is enabled or disabled.</p>
   */
  inline ScheduleState GetState() const { return m_state; }
  inline void SetState(ScheduleState value) {
    m_stateHasBeenSet = true;
    m_state = value;
  }
  inline GetScheduleResult& WithState(ScheduleState value) {
    SetState(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The schedule target.</p>
   */
  inline const Target& GetTarget() const { return m_target; }
  template <typename TargetT = Target>
  void SetTarget(TargetT&& value) {
    m_targetHasBeenSet = true;
    m_target = std::forward<TargetT>(value);
  }
  template <typename TargetT = Target>
  GetScheduleResult& WithTarget(TargetT&& value) {
    SetTarget(std::forward<TargetT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::String& GetRequestId() const { return m_requestId; }
  template <typename RequestIdT = Aws::String>
  void SetRequestId(RequestIdT&& value) {
    m_requestIdHasBeenSet = true;
    m_requestId = std::forward<RequestIdT>(value);
  }
  template <typename RequestIdT = Aws::String>
  GetScheduleResult& WithRequestId(RequestIdT&& value) {
    SetRequestId(std::forward<RequestIdT>(value));
    return *this;
  }
  ///@}
 private:
  ActionAfterCompletion m_actionAfterCompletion{ActionAfterCompletion::NOT_SET};
  bool m_actionAfterCompletionHasBeenSet = false;

  Aws::String m_arn;
  bool m_arnHasBeenSet = false;

  Aws::Utils::DateTime m_creationDate{};
  bool m_creationDateHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  Aws::Utils::DateTime m_endDate{};
  bool m_endDateHasBeenSet = false;

  FlexibleTimeWindow m_flexibleTimeWindow;
  bool m_flexibleTimeWindowHasBeenSet = false;

  Aws::String m_groupName;
  bool m_groupNameHasBeenSet = false;

  Aws::String m_kmsKeyArn;
  bool m_kmsKeyArnHasBeenSet = false;

  Aws::Utils::DateTime m_lastModificationDate{};
  bool m_lastModificationDateHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_scheduleExpression;
  bool m_scheduleExpressionHasBeenSet = false;

  Aws::String m_scheduleExpressionTimezone;
  bool m_scheduleExpressionTimezoneHasBeenSet = false;

  Aws::Utils::DateTime m_startDate{};
  bool m_startDateHasBeenSet = false;

  ScheduleState m_state{ScheduleState::NOT_SET};
  bool m_stateHasBeenSet = false;

  Target m_target;
  bool m_targetHasBeenSet = false;

  Aws::String m_requestId;
  bool m_requestIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace Scheduler
}  // namespace Aws
