﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/medialive/MediaLiveRequest.h>
#include <aws/medialive/MediaLive_EXPORTS.h>
#include <aws/medialive/model/NodeInterfaceMappingCreateRequest.h>
#include <aws/medialive/model/NodeRole.h>

#include <utility>

namespace Aws {
namespace MediaLive {
namespace Model {

/**
 * A request to create a node<p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/CreateNodeRequest">AWS
 * API Reference</a></p>
 */
class CreateNodeRequest : public MediaLiveRequest {
 public:
  AWS_MEDIALIVE_API CreateNodeRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateNode"; }

  AWS_MEDIALIVE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * The ID of the cluster.
   */
  inline const Aws::String& GetClusterId() const { return m_clusterId; }
  inline bool ClusterIdHasBeenSet() const { return m_clusterIdHasBeenSet; }
  template <typename ClusterIdT = Aws::String>
  void SetClusterId(ClusterIdT&& value) {
    m_clusterIdHasBeenSet = true;
    m_clusterId = std::forward<ClusterIdT>(value);
  }
  template <typename ClusterIdT = Aws::String>
  CreateNodeRequest& WithClusterId(ClusterIdT&& value) {
    SetClusterId(std::forward<ClusterIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * The user-specified name of the Node to be created.
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateNodeRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * Documentation update needed
   */
  inline const Aws::Vector<NodeInterfaceMappingCreateRequest>& GetNodeInterfaceMappings() const { return m_nodeInterfaceMappings; }
  inline bool NodeInterfaceMappingsHasBeenSet() const { return m_nodeInterfaceMappingsHasBeenSet; }
  template <typename NodeInterfaceMappingsT = Aws::Vector<NodeInterfaceMappingCreateRequest>>
  void SetNodeInterfaceMappings(NodeInterfaceMappingsT&& value) {
    m_nodeInterfaceMappingsHasBeenSet = true;
    m_nodeInterfaceMappings = std::forward<NodeInterfaceMappingsT>(value);
  }
  template <typename NodeInterfaceMappingsT = Aws::Vector<NodeInterfaceMappingCreateRequest>>
  CreateNodeRequest& WithNodeInterfaceMappings(NodeInterfaceMappingsT&& value) {
    SetNodeInterfaceMappings(std::forward<NodeInterfaceMappingsT>(value));
    return *this;
  }
  template <typename NodeInterfaceMappingsT = NodeInterfaceMappingCreateRequest>
  CreateNodeRequest& AddNodeInterfaceMappings(NodeInterfaceMappingsT&& value) {
    m_nodeInterfaceMappingsHasBeenSet = true;
    m_nodeInterfaceMappings.emplace_back(std::forward<NodeInterfaceMappingsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * An ID that you assign to a create request. This ID ensures idempotency when
   * creating resources.
   */
  inline const Aws::String& GetRequestId() const { return m_requestId; }
  inline bool RequestIdHasBeenSet() const { return m_requestIdHasBeenSet; }
  template <typename RequestIdT = Aws::String>
  void SetRequestId(RequestIdT&& value) {
    m_requestIdHasBeenSet = true;
    m_requestId = std::forward<RequestIdT>(value);
  }
  template <typename RequestIdT = Aws::String>
  CreateNodeRequest& WithRequestId(RequestIdT&& value) {
    SetRequestId(std::forward<RequestIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * The initial role of the Node in the Cluster. ACTIVE means the Node is available
   * for encoding. BACKUP means the Node is a redundant Node and might get used if an
   * ACTIVE Node fails.
   */
  inline NodeRole GetRole() const { return m_role; }
  inline bool RoleHasBeenSet() const { return m_roleHasBeenSet; }
  inline void SetRole(NodeRole value) {
    m_roleHasBeenSet = true;
    m_role = value;
  }
  inline CreateNodeRequest& WithRole(NodeRole value) {
    SetRole(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * A collection of key-value pairs.
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateNodeRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateNodeRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clusterId;
  bool m_clusterIdHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::Vector<NodeInterfaceMappingCreateRequest> m_nodeInterfaceMappings;
  bool m_nodeInterfaceMappingsHasBeenSet = false;

  Aws::String m_requestId{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_requestIdHasBeenSet = true;

  NodeRole m_role{NodeRole::NOT_SET};
  bool m_roleHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace MediaLive
}  // namespace Aws
