﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/events/CloudWatchEventsRequest.h>
#include <aws/events/CloudWatchEvents_EXPORTS.h>

#include <utility>

namespace Aws {
namespace CloudWatchEvents {
namespace Model {

/**
 */
class TestEventPatternRequest : public CloudWatchEventsRequest {
 public:
  AWS_CLOUDWATCHEVENTS_API TestEventPatternRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "TestEventPattern"; }

  AWS_CLOUDWATCHEVENTS_API Aws::String SerializePayload() const override;

  AWS_CLOUDWATCHEVENTS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The event pattern. For more information, see <a
   * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eventbridge-and-event-patterns.html">Events
   * and Event Patterns</a> in the <i>Amazon EventBridge User Guide</i>.</p>
   */
  inline const Aws::String& GetEventPattern() const { return m_eventPattern; }
  inline bool EventPatternHasBeenSet() const { return m_eventPatternHasBeenSet; }
  template <typename EventPatternT = Aws::String>
  void SetEventPattern(EventPatternT&& value) {
    m_eventPatternHasBeenSet = true;
    m_eventPattern = std::forward<EventPatternT>(value);
  }
  template <typename EventPatternT = Aws::String>
  TestEventPatternRequest& WithEventPattern(EventPatternT&& value) {
    SetEventPattern(std::forward<EventPatternT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The event, in JSON format, to test against the event pattern. The JSON must
   * follow the format specified in <a
   * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/aws-events.html">Amazon
   * Web Services Events</a>, and the following fields are mandatory:</p> <ul> <li>
   * <p> <code>id</code> </p> </li> <li> <p> <code>account</code> </p> </li> <li> <p>
   * <code>source</code> </p> </li> <li> <p> <code>time</code> </p> </li> <li> <p>
   * <code>region</code> </p> </li> <li> <p> <code>resources</code> </p> </li> <li>
   * <p> <code>detail-type</code> </p> </li> </ul>
   */
  inline const Aws::String& GetEvent() const { return m_event; }
  inline bool EventHasBeenSet() const { return m_eventHasBeenSet; }
  template <typename EventT = Aws::String>
  void SetEvent(EventT&& value) {
    m_eventHasBeenSet = true;
    m_event = std::forward<EventT>(value);
  }
  template <typename EventT = Aws::String>
  TestEventPatternRequest& WithEvent(EventT&& value) {
    SetEvent(std::forward<EventT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_eventPattern;
  bool m_eventPatternHasBeenSet = false;

  Aws::String m_event;
  bool m_eventHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudWatchEvents
}  // namespace Aws
