﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/docdb/DocDBRequest.h>
#include <aws/docdb/DocDB_EXPORTS.h>
#include <aws/docdb/model/ServerlessV2ScalingConfiguration.h>
#include <aws/docdb/model/Tag.h>

#include <utility>

namespace Aws {
namespace DocDB {
namespace Model {

/**
 * <p>Represents the input to <a>CreateDBCluster</a>. </p><p><h3>See Also:</h3>
 * <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/docdb-2014-10-31/CreateDBClusterMessage">AWS
 * API Reference</a></p>
 */
class CreateDBClusterRequest : public DocDBRequest {
 public:
  AWS_DOCDB_API CreateDBClusterRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateDBCluster"; }

  AWS_DOCDB_API Aws::String SerializePayload() const override;

 protected:
  AWS_DOCDB_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>A list of Amazon EC2 Availability Zones that instances in the cluster can be
   * created in.</p>
   */
  inline const Aws::Vector<Aws::String>& GetAvailabilityZones() const { return m_availabilityZones; }
  inline bool AvailabilityZonesHasBeenSet() const { return m_availabilityZonesHasBeenSet; }
  template <typename AvailabilityZonesT = Aws::Vector<Aws::String>>
  void SetAvailabilityZones(AvailabilityZonesT&& value) {
    m_availabilityZonesHasBeenSet = true;
    m_availabilityZones = std::forward<AvailabilityZonesT>(value);
  }
  template <typename AvailabilityZonesT = Aws::Vector<Aws::String>>
  CreateDBClusterRequest& WithAvailabilityZones(AvailabilityZonesT&& value) {
    SetAvailabilityZones(std::forward<AvailabilityZonesT>(value));
    return *this;
  }
  template <typename AvailabilityZonesT = Aws::String>
  CreateDBClusterRequest& AddAvailabilityZones(AvailabilityZonesT&& value) {
    m_availabilityZonesHasBeenSet = true;
    m_availabilityZones.emplace_back(std::forward<AvailabilityZonesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of days for which automated backups are retained. You must specify
   * a minimum value of 1.</p> <p>Default: 1</p> <p>Constraints:</p> <ul> <li>
   * <p>Must be a value from 1 to 35.</p> </li> </ul>
   */
  inline int GetBackupRetentionPeriod() const { return m_backupRetentionPeriod; }
  inline bool BackupRetentionPeriodHasBeenSet() const { return m_backupRetentionPeriodHasBeenSet; }
  inline void SetBackupRetentionPeriod(int value) {
    m_backupRetentionPeriodHasBeenSet = true;
    m_backupRetentionPeriod = value;
  }
  inline CreateDBClusterRequest& WithBackupRetentionPeriod(int value) {
    SetBackupRetentionPeriod(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The cluster identifier. This parameter is stored as a lowercase string.</p>
   * <p>Constraints:</p> <ul> <li> <p>Must contain from 1 to 63 letters, numbers, or
   * hyphens. </p> </li> <li> <p>The first character must be a letter.</p> </li> <li>
   * <p>Cannot end with a hyphen or contain two consecutive hyphens. </p> </li> </ul>
   * <p>Example: <code>my-cluster</code> </p>
   */
  inline const Aws::String& GetDBClusterIdentifier() const { return m_dBClusterIdentifier; }
  inline bool DBClusterIdentifierHasBeenSet() const { return m_dBClusterIdentifierHasBeenSet; }
  template <typename DBClusterIdentifierT = Aws::String>
  void SetDBClusterIdentifier(DBClusterIdentifierT&& value) {
    m_dBClusterIdentifierHasBeenSet = true;
    m_dBClusterIdentifier = std::forward<DBClusterIdentifierT>(value);
  }
  template <typename DBClusterIdentifierT = Aws::String>
  CreateDBClusterRequest& WithDBClusterIdentifier(DBClusterIdentifierT&& value) {
    SetDBClusterIdentifier(std::forward<DBClusterIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the cluster parameter group to associate with this cluster.</p>
   */
  inline const Aws::String& GetDBClusterParameterGroupName() const { return m_dBClusterParameterGroupName; }
  inline bool DBClusterParameterGroupNameHasBeenSet() const { return m_dBClusterParameterGroupNameHasBeenSet; }
  template <typename DBClusterParameterGroupNameT = Aws::String>
  void SetDBClusterParameterGroupName(DBClusterParameterGroupNameT&& value) {
    m_dBClusterParameterGroupNameHasBeenSet = true;
    m_dBClusterParameterGroupName = std::forward<DBClusterParameterGroupNameT>(value);
  }
  template <typename DBClusterParameterGroupNameT = Aws::String>
  CreateDBClusterRequest& WithDBClusterParameterGroupName(DBClusterParameterGroupNameT&& value) {
    SetDBClusterParameterGroupName(std::forward<DBClusterParameterGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of EC2 VPC security groups to associate with this cluster. </p>
   */
  inline const Aws::Vector<Aws::String>& GetVpcSecurityGroupIds() const { return m_vpcSecurityGroupIds; }
  inline bool VpcSecurityGroupIdsHasBeenSet() const { return m_vpcSecurityGroupIdsHasBeenSet; }
  template <typename VpcSecurityGroupIdsT = Aws::Vector<Aws::String>>
  void SetVpcSecurityGroupIds(VpcSecurityGroupIdsT&& value) {
    m_vpcSecurityGroupIdsHasBeenSet = true;
    m_vpcSecurityGroupIds = std::forward<VpcSecurityGroupIdsT>(value);
  }
  template <typename VpcSecurityGroupIdsT = Aws::Vector<Aws::String>>
  CreateDBClusterRequest& WithVpcSecurityGroupIds(VpcSecurityGroupIdsT&& value) {
    SetVpcSecurityGroupIds(std::forward<VpcSecurityGroupIdsT>(value));
    return *this;
  }
  template <typename VpcSecurityGroupIdsT = Aws::String>
  CreateDBClusterRequest& AddVpcSecurityGroupIds(VpcSecurityGroupIdsT&& value) {
    m_vpcSecurityGroupIdsHasBeenSet = true;
    m_vpcSecurityGroupIds.emplace_back(std::forward<VpcSecurityGroupIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A subnet group to associate with this cluster.</p> <p>Constraints: Must match
   * the name of an existing <code>DBSubnetGroup</code>. Must not be default.</p>
   * <p>Example: <code>mySubnetgroup</code> </p>
   */
  inline const Aws::String& GetDBSubnetGroupName() const { return m_dBSubnetGroupName; }
  inline bool DBSubnetGroupNameHasBeenSet() const { return m_dBSubnetGroupNameHasBeenSet; }
  template <typename DBSubnetGroupNameT = Aws::String>
  void SetDBSubnetGroupName(DBSubnetGroupNameT&& value) {
    m_dBSubnetGroupNameHasBeenSet = true;
    m_dBSubnetGroupName = std::forward<DBSubnetGroupNameT>(value);
  }
  template <typename DBSubnetGroupNameT = Aws::String>
  CreateDBClusterRequest& WithDBSubnetGroupName(DBSubnetGroupNameT&& value) {
    SetDBSubnetGroupName(std::forward<DBSubnetGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the database engine to be used for this cluster.</p> <p>Valid
   * values: <code>docdb</code> </p>
   */
  inline const Aws::String& GetEngine() const { return m_engine; }
  inline bool EngineHasBeenSet() const { return m_engineHasBeenSet; }
  template <typename EngineT = Aws::String>
  void SetEngine(EngineT&& value) {
    m_engineHasBeenSet = true;
    m_engine = std::forward<EngineT>(value);
  }
  template <typename EngineT = Aws::String>
  CreateDBClusterRequest& WithEngine(EngineT&& value) {
    SetEngine(std::forward<EngineT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version number of the database engine to use. The
   * <code>--engine-version</code> will default to the latest major engine version.
   * For production workloads, we recommend explicitly declaring this parameter with
   * the intended major engine version.</p>
   */
  inline const Aws::String& GetEngineVersion() const { return m_engineVersion; }
  inline bool EngineVersionHasBeenSet() const { return m_engineVersionHasBeenSet; }
  template <typename EngineVersionT = Aws::String>
  void SetEngineVersion(EngineVersionT&& value) {
    m_engineVersionHasBeenSet = true;
    m_engineVersion = std::forward<EngineVersionT>(value);
  }
  template <typename EngineVersionT = Aws::String>
  CreateDBClusterRequest& WithEngineVersion(EngineVersionT&& value) {
    SetEngineVersion(std::forward<EngineVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The port number on which the instances in the cluster accept connections.</p>
   */
  inline int GetPort() const { return m_port; }
  inline bool PortHasBeenSet() const { return m_portHasBeenSet; }
  inline void SetPort(int value) {
    m_portHasBeenSet = true;
    m_port = value;
  }
  inline CreateDBClusterRequest& WithPort(int value) {
    SetPort(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the master user for the cluster.</p> <p>Constraints:</p> <ul>
   * <li> <p>Must be from 1 to 63 letters or numbers.</p> </li> <li> <p>The first
   * character must be a letter.</p> </li> <li> <p>Cannot be a reserved word for the
   * chosen database engine. </p> </li> </ul>
   */
  inline const Aws::String& GetMasterUsername() const { return m_masterUsername; }
  inline bool MasterUsernameHasBeenSet() const { return m_masterUsernameHasBeenSet; }
  template <typename MasterUsernameT = Aws::String>
  void SetMasterUsername(MasterUsernameT&& value) {
    m_masterUsernameHasBeenSet = true;
    m_masterUsername = std::forward<MasterUsernameT>(value);
  }
  template <typename MasterUsernameT = Aws::String>
  CreateDBClusterRequest& WithMasterUsername(MasterUsernameT&& value) {
    SetMasterUsername(std::forward<MasterUsernameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The password for the master database user. This password can contain any
   * printable ASCII character except forward slash (/), double quote ("), or the
   * "at" symbol (@).</p> <p>Constraints: Must contain from 8 to 100 characters.</p>
   */
  inline const Aws::String& GetMasterUserPassword() const { return m_masterUserPassword; }
  inline bool MasterUserPasswordHasBeenSet() const { return m_masterUserPasswordHasBeenSet; }
  template <typename MasterUserPasswordT = Aws::String>
  void SetMasterUserPassword(MasterUserPasswordT&& value) {
    m_masterUserPasswordHasBeenSet = true;
    m_masterUserPassword = std::forward<MasterUserPasswordT>(value);
  }
  template <typename MasterUserPasswordT = Aws::String>
  CreateDBClusterRequest& WithMasterUserPassword(MasterUserPasswordT&& value) {
    SetMasterUserPassword(std::forward<MasterUserPasswordT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The daily time range during which automated backups are created if automated
   * backups are enabled using the <code>BackupRetentionPeriod</code> parameter. </p>
   * <p>The default is a 30-minute window selected at random from an 8-hour block of
   * time for each Amazon Web Services Region. </p> <p>Constraints:</p> <ul> <li>
   * <p>Must be in the format <code>hh24:mi-hh24:mi</code>.</p> </li> <li> <p>Must be
   * in Universal Coordinated Time (UTC).</p> </li> <li> <p>Must not conflict with
   * the preferred maintenance window. </p> </li> <li> <p>Must be at least 30
   * minutes.</p> </li> </ul>
   */
  inline const Aws::String& GetPreferredBackupWindow() const { return m_preferredBackupWindow; }
  inline bool PreferredBackupWindowHasBeenSet() const { return m_preferredBackupWindowHasBeenSet; }
  template <typename PreferredBackupWindowT = Aws::String>
  void SetPreferredBackupWindow(PreferredBackupWindowT&& value) {
    m_preferredBackupWindowHasBeenSet = true;
    m_preferredBackupWindow = std::forward<PreferredBackupWindowT>(value);
  }
  template <typename PreferredBackupWindowT = Aws::String>
  CreateDBClusterRequest& WithPreferredBackupWindow(PreferredBackupWindowT&& value) {
    SetPreferredBackupWindow(std::forward<PreferredBackupWindowT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The weekly time range during which system maintenance can occur, in Universal
   * Coordinated Time (UTC).</p> <p>Format: <code>ddd:hh24:mi-ddd:hh24:mi</code> </p>
   * <p>The default is a 30-minute window selected at random from an 8-hour block of
   * time for each Amazon Web Services Region, occurring on a random day of the
   * week.</p> <p>Valid days: Mon, Tue, Wed, Thu, Fri, Sat, Sun</p> <p>Constraints:
   * Minimum 30-minute window.</p>
   */
  inline const Aws::String& GetPreferredMaintenanceWindow() const { return m_preferredMaintenanceWindow; }
  inline bool PreferredMaintenanceWindowHasBeenSet() const { return m_preferredMaintenanceWindowHasBeenSet; }
  template <typename PreferredMaintenanceWindowT = Aws::String>
  void SetPreferredMaintenanceWindow(PreferredMaintenanceWindowT&& value) {
    m_preferredMaintenanceWindowHasBeenSet = true;
    m_preferredMaintenanceWindow = std::forward<PreferredMaintenanceWindowT>(value);
  }
  template <typename PreferredMaintenanceWindowT = Aws::String>
  CreateDBClusterRequest& WithPreferredMaintenanceWindow(PreferredMaintenanceWindowT&& value) {
    SetPreferredMaintenanceWindow(std::forward<PreferredMaintenanceWindowT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags to be assigned to the cluster.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateDBClusterRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateDBClusterRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether the cluster is encrypted.</p>
   */
  inline bool GetStorageEncrypted() const { return m_storageEncrypted; }
  inline bool StorageEncryptedHasBeenSet() const { return m_storageEncryptedHasBeenSet; }
  inline void SetStorageEncrypted(bool value) {
    m_storageEncryptedHasBeenSet = true;
    m_storageEncrypted = value;
  }
  inline CreateDBClusterRequest& WithStorageEncrypted(bool value) {
    SetStorageEncrypted(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The KMS key identifier for an encrypted cluster.</p> <p>The KMS key
   * identifier is the Amazon Resource Name (ARN) for the KMS encryption key. If you
   * are creating a cluster using the same Amazon Web Services account that owns the
   * KMS encryption key that is used to encrypt the new cluster, you can use the KMS
   * key alias instead of the ARN for the KMS encryption key.</p> <p>If an encryption
   * key is not specified in <code>KmsKeyId</code>: </p> <ul> <li> <p>If the
   * <code>StorageEncrypted</code> parameter is <code>true</code>, Amazon DocumentDB
   * uses your default encryption key. </p> </li> </ul> <p>KMS creates the default
   * encryption key for your Amazon Web Services account. Your Amazon Web Services
   * account has a different default encryption key for each Amazon Web Services
   * Regions.</p>
   */
  inline const Aws::String& GetKmsKeyId() const { return m_kmsKeyId; }
  inline bool KmsKeyIdHasBeenSet() const { return m_kmsKeyIdHasBeenSet; }
  template <typename KmsKeyIdT = Aws::String>
  void SetKmsKeyId(KmsKeyIdT&& value) {
    m_kmsKeyIdHasBeenSet = true;
    m_kmsKeyId = std::forward<KmsKeyIdT>(value);
  }
  template <typename KmsKeyIdT = Aws::String>
  CreateDBClusterRequest& WithKmsKeyId(KmsKeyIdT&& value) {
    SetKmsKeyId(std::forward<KmsKeyIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Not currently supported. </p>
   */
  inline const Aws::String& GetPreSignedUrl() const { return m_preSignedUrl; }
  inline bool PreSignedUrlHasBeenSet() const { return m_preSignedUrlHasBeenSet; }
  template <typename PreSignedUrlT = Aws::String>
  void SetPreSignedUrl(PreSignedUrlT&& value) {
    m_preSignedUrlHasBeenSet = true;
    m_preSignedUrl = std::forward<PreSignedUrlT>(value);
  }
  template <typename PreSignedUrlT = Aws::String>
  CreateDBClusterRequest& WithPreSignedUrl(PreSignedUrlT&& value) {
    SetPreSignedUrl(std::forward<PreSignedUrlT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of log types that need to be enabled for exporting to Amazon
   * CloudWatch Logs. You can enable audit logs or profiler logs. For more
   * information, see <a
   * href="https://docs.aws.amazon.com/documentdb/latest/developerguide/event-auditing.html">
   * Auditing Amazon DocumentDB Events</a> and <a
   * href="https://docs.aws.amazon.com/documentdb/latest/developerguide/profiling.html">
   * Profiling Amazon DocumentDB Operations</a>. </p>
   */
  inline const Aws::Vector<Aws::String>& GetEnableCloudwatchLogsExports() const { return m_enableCloudwatchLogsExports; }
  inline bool EnableCloudwatchLogsExportsHasBeenSet() const { return m_enableCloudwatchLogsExportsHasBeenSet; }
  template <typename EnableCloudwatchLogsExportsT = Aws::Vector<Aws::String>>
  void SetEnableCloudwatchLogsExports(EnableCloudwatchLogsExportsT&& value) {
    m_enableCloudwatchLogsExportsHasBeenSet = true;
    m_enableCloudwatchLogsExports = std::forward<EnableCloudwatchLogsExportsT>(value);
  }
  template <typename EnableCloudwatchLogsExportsT = Aws::Vector<Aws::String>>
  CreateDBClusterRequest& WithEnableCloudwatchLogsExports(EnableCloudwatchLogsExportsT&& value) {
    SetEnableCloudwatchLogsExports(std::forward<EnableCloudwatchLogsExportsT>(value));
    return *this;
  }
  template <typename EnableCloudwatchLogsExportsT = Aws::String>
  CreateDBClusterRequest& AddEnableCloudwatchLogsExports(EnableCloudwatchLogsExportsT&& value) {
    m_enableCloudwatchLogsExportsHasBeenSet = true;
    m_enableCloudwatchLogsExports.emplace_back(std::forward<EnableCloudwatchLogsExportsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether this cluster can be deleted. If
   * <code>DeletionProtection</code> is enabled, the cluster cannot be deleted unless
   * it is modified and <code>DeletionProtection</code> is disabled.
   * <code>DeletionProtection</code> protects clusters from being accidentally
   * deleted.</p>
   */
  inline bool GetDeletionProtection() const { return m_deletionProtection; }
  inline bool DeletionProtectionHasBeenSet() const { return m_deletionProtectionHasBeenSet; }
  inline void SetDeletionProtection(bool value) {
    m_deletionProtectionHasBeenSet = true;
    m_deletionProtection = value;
  }
  inline CreateDBClusterRequest& WithDeletionProtection(bool value) {
    SetDeletionProtection(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The cluster identifier of the new global cluster.</p>
   */
  inline const Aws::String& GetGlobalClusterIdentifier() const { return m_globalClusterIdentifier; }
  inline bool GlobalClusterIdentifierHasBeenSet() const { return m_globalClusterIdentifierHasBeenSet; }
  template <typename GlobalClusterIdentifierT = Aws::String>
  void SetGlobalClusterIdentifier(GlobalClusterIdentifierT&& value) {
    m_globalClusterIdentifierHasBeenSet = true;
    m_globalClusterIdentifier = std::forward<GlobalClusterIdentifierT>(value);
  }
  template <typename GlobalClusterIdentifierT = Aws::String>
  CreateDBClusterRequest& WithGlobalClusterIdentifier(GlobalClusterIdentifierT&& value) {
    SetGlobalClusterIdentifier(std::forward<GlobalClusterIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The storage type to associate with the DB cluster.</p> <p>For information on
   * storage types for Amazon DocumentDB clusters, see Cluster storage configurations
   * in the <i>Amazon DocumentDB Developer Guide</i>.</p> <p>Valid values for storage
   * type - <code>standard | iopt1</code> </p> <p>Default value is <code>standard
   * </code> </p>  <p>When you create an Amazon DocumentDB cluster with the
   * storage type set to <code>iopt1</code>, the storage type is returned in the
   * response. The storage type isn't returned when you set it to
   * <code>standard</code>.</p>
   */
  inline const Aws::String& GetStorageType() const { return m_storageType; }
  inline bool StorageTypeHasBeenSet() const { return m_storageTypeHasBeenSet; }
  template <typename StorageTypeT = Aws::String>
  void SetStorageType(StorageTypeT&& value) {
    m_storageTypeHasBeenSet = true;
    m_storageType = std::forward<StorageTypeT>(value);
  }
  template <typename StorageTypeT = Aws::String>
  CreateDBClusterRequest& WithStorageType(StorageTypeT&& value) {
    SetStorageType(std::forward<StorageTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Contains the scaling configuration of an Amazon DocumentDB Serverless
   * cluster.</p>
   */
  inline const ServerlessV2ScalingConfiguration& GetServerlessV2ScalingConfiguration() const { return m_serverlessV2ScalingConfiguration; }
  inline bool ServerlessV2ScalingConfigurationHasBeenSet() const { return m_serverlessV2ScalingConfigurationHasBeenSet; }
  template <typename ServerlessV2ScalingConfigurationT = ServerlessV2ScalingConfiguration>
  void SetServerlessV2ScalingConfiguration(ServerlessV2ScalingConfigurationT&& value) {
    m_serverlessV2ScalingConfigurationHasBeenSet = true;
    m_serverlessV2ScalingConfiguration = std::forward<ServerlessV2ScalingConfigurationT>(value);
  }
  template <typename ServerlessV2ScalingConfigurationT = ServerlessV2ScalingConfiguration>
  CreateDBClusterRequest& WithServerlessV2ScalingConfiguration(ServerlessV2ScalingConfigurationT&& value) {
    SetServerlessV2ScalingConfiguration(std::forward<ServerlessV2ScalingConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether to manage the master user password with Amazon Web Services
   * Secrets Manager.</p> <p>Constraint: You can't manage the master user password
   * with Amazon Web Services Secrets Manager if <code>MasterUserPassword</code> is
   * specified.</p>
   */
  inline bool GetManageMasterUserPassword() const { return m_manageMasterUserPassword; }
  inline bool ManageMasterUserPasswordHasBeenSet() const { return m_manageMasterUserPasswordHasBeenSet; }
  inline void SetManageMasterUserPassword(bool value) {
    m_manageMasterUserPasswordHasBeenSet = true;
    m_manageMasterUserPassword = value;
  }
  inline CreateDBClusterRequest& WithManageMasterUserPassword(bool value) {
    SetManageMasterUserPassword(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Web Services KMS key identifier to encrypt a secret that is
   * automatically generated and managed in Amazon Web Services Secrets Manager. This
   * setting is valid only if the master user password is managed by Amazon
   * DocumentDB in Amazon Web Services Secrets Manager for the DB cluster.</p> <p>The
   * Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or
   * alias name for the KMS key. To use a KMS key in a different Amazon Web Services
   * account, specify the key ARN or alias ARN.</p> <p>If you don't specify
   * <code>MasterUserSecretKmsKeyId</code>, then the <code>aws/secretsmanager</code>
   * KMS key is used to encrypt the secret. If the secret is in a different Amazon
   * Web Services account, then you can't use the <code>aws/secretsmanager</code> KMS
   * key to encrypt the secret, and you must use a customer managed KMS key.</p>
   * <p>There is a default KMS key for your Amazon Web Services account. Your Amazon
   * Web Services account has a different default KMS key for each Amazon Web
   * Services Region.</p>
   */
  inline const Aws::String& GetMasterUserSecretKmsKeyId() const { return m_masterUserSecretKmsKeyId; }
  inline bool MasterUserSecretKmsKeyIdHasBeenSet() const { return m_masterUserSecretKmsKeyIdHasBeenSet; }
  template <typename MasterUserSecretKmsKeyIdT = Aws::String>
  void SetMasterUserSecretKmsKeyId(MasterUserSecretKmsKeyIdT&& value) {
    m_masterUserSecretKmsKeyIdHasBeenSet = true;
    m_masterUserSecretKmsKeyId = std::forward<MasterUserSecretKmsKeyIdT>(value);
  }
  template <typename MasterUserSecretKmsKeyIdT = Aws::String>
  CreateDBClusterRequest& WithMasterUserSecretKmsKeyId(MasterUserSecretKmsKeyIdT&& value) {
    SetMasterUserSecretKmsKeyId(std::forward<MasterUserSecretKmsKeyIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The network type of the cluster.</p> <p>The network type is determined by the
   * <code>DBSubnetGroup</code> specified for the cluster. A
   * <code>DBSubnetGroup</code> can support only the IPv4 protocol or the IPv4 and
   * the IPv6 protocols (<code>DUAL</code>).</p> <p>For more information, see <a
   * href="https://docs.aws.amazon.com/documentdb/latest/developerguide/vpc-clusters.html">DocumentDB
   * clusters in a VPC</a> in the Amazon DocumentDB Developer Guide.</p> <p>Valid
   * Values: <code>IPV4</code> | <code>DUAL</code> </p>
   */
  inline const Aws::String& GetNetworkType() const { return m_networkType; }
  inline bool NetworkTypeHasBeenSet() const { return m_networkTypeHasBeenSet; }
  template <typename NetworkTypeT = Aws::String>
  void SetNetworkType(NetworkTypeT&& value) {
    m_networkTypeHasBeenSet = true;
    m_networkType = std::forward<NetworkTypeT>(value);
  }
  template <typename NetworkTypeT = Aws::String>
  CreateDBClusterRequest& WithNetworkType(NetworkTypeT&& value) {
    SetNetworkType(std::forward<NetworkTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * If SourceRegion is specified, SDKs will generate pre-signed URLs and populate the pre-signed URL field.
   */
  inline const Aws::String& GetSourceRegion() const { return m_sourceRegion; }
  inline bool SourceRegionHasBeenSet() const { return m_sourceRegionHasBeenSet; }
  template <typename SourceRegionT = Aws::String>
  void SetSourceRegion(SourceRegionT&& value) {
    m_sourceRegionHasBeenSet = true;
    m_sourceRegion = std::forward<SourceRegionT>(value);
  }
  template <typename SourceRegionT = Aws::String>
  CreateDBClusterRequest& WithSourceRegion(SourceRegionT&& value) {
    SetSourceRegion(std::forward<SourceRegionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Aws::String> m_availabilityZones;
  bool m_availabilityZonesHasBeenSet = false;

  int m_backupRetentionPeriod{0};
  bool m_backupRetentionPeriodHasBeenSet = false;

  Aws::String m_dBClusterIdentifier;
  bool m_dBClusterIdentifierHasBeenSet = false;

  Aws::String m_dBClusterParameterGroupName;
  bool m_dBClusterParameterGroupNameHasBeenSet = false;

  Aws::Vector<Aws::String> m_vpcSecurityGroupIds;
  bool m_vpcSecurityGroupIdsHasBeenSet = false;

  Aws::String m_dBSubnetGroupName;
  bool m_dBSubnetGroupNameHasBeenSet = false;

  Aws::String m_engine;
  bool m_engineHasBeenSet = false;

  Aws::String m_engineVersion;
  bool m_engineVersionHasBeenSet = false;

  int m_port{0};
  bool m_portHasBeenSet = false;

  Aws::String m_masterUsername;
  bool m_masterUsernameHasBeenSet = false;

  Aws::String m_masterUserPassword;
  bool m_masterUserPasswordHasBeenSet = false;

  Aws::String m_preferredBackupWindow;
  bool m_preferredBackupWindowHasBeenSet = false;

  Aws::String m_preferredMaintenanceWindow;
  bool m_preferredMaintenanceWindowHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;

  bool m_storageEncrypted{false};
  bool m_storageEncryptedHasBeenSet = false;

  Aws::String m_kmsKeyId;
  bool m_kmsKeyIdHasBeenSet = false;

  Aws::String m_preSignedUrl;
  bool m_preSignedUrlHasBeenSet = false;

  Aws::Vector<Aws::String> m_enableCloudwatchLogsExports;
  bool m_enableCloudwatchLogsExportsHasBeenSet = false;

  bool m_deletionProtection{false};
  bool m_deletionProtectionHasBeenSet = false;

  Aws::String m_globalClusterIdentifier;
  bool m_globalClusterIdentifierHasBeenSet = false;

  Aws::String m_storageType;
  bool m_storageTypeHasBeenSet = false;

  ServerlessV2ScalingConfiguration m_serverlessV2ScalingConfiguration;
  bool m_serverlessV2ScalingConfigurationHasBeenSet = false;

  bool m_manageMasterUserPassword{false};
  bool m_manageMasterUserPasswordHasBeenSet = false;

  Aws::String m_masterUserSecretKmsKeyId;
  bool m_masterUserSecretKmsKeyIdHasBeenSet = false;

  Aws::String m_networkType;
  bool m_networkTypeHasBeenSet = false;

  Aws::String m_sourceRegion;
  bool m_sourceRegionHasBeenSet = false;
};

}  // namespace Model
}  // namespace DocDB
}  // namespace Aws
