#!/usr/bin/env python
########################################################
#
# Script to generate cpu hotspot stats from an nsys trace file
#
########################################################
import fnmatch
import logging
import os
import re
import shutil
import subprocess
import sys
import threading
import time
import traceback
from typing import List, Optional

import typer
import uvicorn
from pydantic.v1 import BaseModel

sys.path.extend(["..", "."])  # https://stackoverflow.com/a/45874916/1281963
from nsys_cpu_stats import trace_utils as tu
from nsys_cpu_stats.cpuframeinfo import CPUFrameInfo, HotspotRange, HotspotReportThreading, HotspotMetricType, TimeSlice
from nsys_cpu_stats.cputhreadinfo import CPUThreadingInfo
from nsys_cpu_stats.report_exporter import ReportExporter, DatabaseIds
from nsys_cpu_stats.trace_loader import TraceLoaderRegions
from nsys_cpu_stats.trace_progress_indicator import g_progress_indicator as progress

if os.name == 'nt':
    g_is_windows = True
    g_nsys_cli_executable_name = 'nsys.exe'
    g_nsys_gui_executable_name = 'nsys-ui.exe'
    g_nsys_sub_path = 'target-windows-x64'
else:  # os.name != 'nt'
    g_is_windows = False
    g_nsys_cli_executable_name = 'nsys'
    g_nsys_gui_executable_name = 'nsys-ui'
    g_nsys_sub_path = 'target-linux-x64'

cli = typer.Typer()
logger = logging.getLogger(__name__)

logging.getLogger("urllib3").setLevel(logging.WARN)

g_logger_stdout = os.getenv('PNR_LOGGER_STDOUT', 'True').lower() in ('true', '1', 't')
g_logger_severity = os.getenv('PNR_LOGGER_SEVERITY', 'DEBUG')

if g_logger_stdout:
    # Format: https://docs.python.org/3/library/logging.html#logrecord-attributes
    logging.basicConfig(level=g_logger_severity,
                        format='[%(levelname)-7s][%(asctime)s] %(message)s',
                        handlers=[logging.StreamHandler(sys.stdout)])

g_enable_multithreading = os.getenv('PNR_ENABLE_MULTITHREADING', 'True').lower() in ('true', '1', 't')
# TODO: use as logger setting
g_quiet = os.getenv('PNR_QUIET', 'True').lower() in ('true', '1', 't')
g_do_sql = os.getenv('PNR_DO_SQL', 'True').lower() in ('true', '1', 't')
g_do_sql_force = os.getenv('PNR_DO_SQL_FORCE', 'False').lower() in ('true', '1', 't')
g_do_threading_info = os.getenv('PNR_DO_THREADING_INFO', 'True').lower() in ('true', '1', 't')
g_do_hotspot_info = os.getenv('PNR_DO_HOTSPOT_INFO', 'True').lower() in ('true', '1', 't')
g_detect_p_cores = os.getenv('PNR_DETECT_P_CORES', 'True').lower() in ('true', '1', 't')
g_clean_output_path = os.getenv('PNR_CLEAN_OUTPUT_PATH', 'False').lower() in ('true', '1', 't')
g_nsys_version_string = None


def __find_file_in_path(pattern, path, sub_path: Optional[str] = None) -> Optional[List[str]]:
    result = []
    for root, dirs, files in os.walk(path):
        if sub_path and sub_path not in root:
            continue
        for name in files:
            if fnmatch.fnmatch(name, pattern):
                result.append(os.path.join(root, name))
    return result


def __find_file_or_exit(exe_string, exe_path, sub_path: Optional[str] = None) -> str:
    exe = __find_file_in_path(exe_string, exe_path, sub_path)
    if len(exe) == 0:
        logger.error(f"Did not find {exe_string}")
        sys.exit(1)
    return exe[0]

def generate_threading_info(input_sql_filename, start_time, end_time, target_pid, report_output_path: str, meta_info: tu.SourceMetaInfo, report_exporter: ReportExporter):
    cpu_thread_info = CPUThreadingInfo(quiet=g_quiet)
    cpu_thread_info.init_nsys_database(input_sql_filename)

    cpu_config = tu.CPUConfig

    if g_detect_p_cores:
        cpu_config = tu.get_PE_core_counts_from_filename(input_sql_filename)
        if cpu_config:
            logger.info(f"Detected {cpu_config.logical_p_core_count} P cores and {cpu_config.physical_e_core_count} E cores from the filename.")

    cpu_thread_info.process_sqldb(target_pid, None,
                                  start_time_ns=start_time, end_time_ns=end_time,
                                  cpu_config=cpu_config)

    cpu_thread_info.close_nsys_database()

    cpu_thread_info.tp.add_meta_info(meta_info)
    report_exporter.export_threading_analysis(cpu_thread_info.tp.df_dict)


def find_target_pid_and_range(input_sql_filename: str, target_process_name: str):
    cpu_frame_info = CPUFrameInfo()
    cpu_frame_info.init_nsys_database(input_sql_filename)

    target_pid = cpu_frame_info.tp.get_process_pid(target_process_name) if target_process_name else None
    filtered_timeslice_list, start_time_ns, end_time_ns, target_pid = cpu_frame_info.get_filtered_timeslices(None, None,
                                                                                                             target_pid, True)
    return start_time_ns, end_time_ns, target_pid


class GenerateSingleHotspotReportParams(BaseModel):
    input_sql_filename: str
    start_time_ns: Optional[float]
    end_time_ns: Optional[float]
    target_pid: Optional[int]
    filtered_timeslice_list: Optional[List[TimeSlice]]
    report_range: HotspotRange
    report_threading: HotspotReportThreading
    report_metric: set[HotspotMetricType]
    report_frame_count: int
    report_thread_count: Optional[int]
    report_median_frame: bool
    report_info: bool
    report_output_path: str
    report_exporter: Optional[ReportExporter]

    class Config:
        arbitrary_types_allowed = True

    # Type-safe method to create updated instances
    def copy_values(self,
                    input_sql_filename: Optional[str] = None,
                    start_time_ns: Optional[float] = None,
                    end_time_ns: Optional[float] = None,
                    target_pid: Optional[int] = None,
                    filtered_timeslice_list: Optional[List[TimeSlice]] = None,
                    report_range: Optional[HotspotRange] = None,
                    report_threading: Optional[HotspotReportThreading] = None,
                    report_metric: Optional[set[HotspotMetricType]] = None,
                    report_frame_count: Optional[int] = None,
                    report_thread_count: Optional[int] = None,
                    report_median_frame: Optional[bool] = None,
                    report_info: Optional[bool] = None,
                    report_output_path: Optional[str] = None,
                    report_exporter: Optional[ReportExporter] = None) -> "GenerateSingleHotspotReportParams":
        # Create a new Params object with updated values while preserving type safety
        return GenerateSingleHotspotReportParams(
            input_sql_filename=input_sql_filename if input_sql_filename is not None else self.input_sql_filename,
            start_time_ns=start_time_ns if start_time_ns is not None else self.start_time_ns,
            end_time_ns=end_time_ns if end_time_ns is not None else self.end_time_ns,
            target_pid=target_pid if target_pid is not None else self.target_pid,
            filtered_timeslice_list=filtered_timeslice_list if filtered_timeslice_list is not None else self.filtered_timeslice_list,
            report_range=report_range if report_range is not None else self.report_range,
            report_threading=report_threading if report_threading is not None else self.report_threading,
            report_metric=report_metric if report_metric is not None else self.report_metric,
            report_frame_count=report_frame_count if report_frame_count is not None else self.report_frame_count,
            report_thread_count=report_thread_count if report_thread_count is not None else self.report_thread_count,
            report_median_frame=report_median_frame if report_median_frame is not None else self.report_median_frame,
            report_info=report_info if report_info is not None else self.report_info,
            report_output_path=report_output_path if report_output_path is not None else self.report_output_path,
            report_exporter=report_exporter if report_exporter is not None else self.report_exporter
        )


def generate_single_hotspot_report(params: GenerateSingleHotspotReportParams):
    cpu_frame_info = CPUFrameInfo()

    cpu_frame_info.init_nsys_database(params.input_sql_filename)

    cpu_config = tu.CPUConfig
    if g_detect_p_cores:
        cpu_config = tu.get_PE_core_counts_from_filename(params.input_sql_filename)
        if cpu_config:
            logger.info(f"Detected {cpu_config.logical_p_core_count} P cores and {cpu_config.physical_e_core_count} E cores.")

    logger.info("=" * 80)
    logger.info(f"Hotspot Analysis: {str(params.report_range)}, {str(params.report_threading)}")
    logger.info("=" * 80)

    filtered_timeslice_list, start_time_ns, end_time_ns, target_pid = params.filtered_timeslice_list, params.start_time_ns, params.end_time_ns, params.target_pid
    if params.filtered_timeslice_list is None or params.target_pid is None:
        filtered_timeslice_list, start_time_ns, end_time_ns, target_pid = cpu_frame_info.get_filtered_timeslices(
            params.start_time_ns, params.end_time_ns, params.target_pid, g_quiet)

    cpu_frame_info.process_stutter_analysis(filtered_timeslice_list,
                                            start_time_ns, end_time_ns, target_pid,
                                            params.report_range, params.report_threading, params.report_metric,
                                            params.report_frame_count, params.report_thread_count,
                                            params.report_median_frame, params.report_info, cpu_config=cpu_config,
                                            quiet=g_quiet, report_exporter=params.report_exporter)
    cpu_frame_info.close_nsys_database()


def generate_hotspot_info(input_sql_filename: str,
                          start_time: Optional[float] = None,
                          end_time: Optional[float] = None,
                          target_pid: Optional[int] = None,
                          multithreading: bool = False,
                          report_output_path: str = '',
                          report_exporter: Optional[ReportExporter] = None):
    # Get the timeslice list
    cpu_frame_info = CPUFrameInfo()
    cpu_frame_info.init_nsys_database(input_sql_filename)
    filtered_timeslice_list, start_time, end_time, target_pid = cpu_frame_info.get_filtered_timeslices(start_time,
                                                                                                       end_time,
                                                                                                       target_pid,
                                                                                                       g_quiet)
    # Would be nice if this was specific to the current PID, which might not support CPU frametimes
    cpu_frametimes_available = cpu_frame_info.tp.is_region_supported(TraceLoaderRegions.CPU_FRAMETIMES)

    thread_count = None
    frame_count = 5
    periodic_frame_count = 10
    capture_median_frame = True
    # Some of the Capture info, such as process name etc, used to be captured each time and when MT,
    # could cause collisions when trying to write to it at the same time. We now just output once.
    # TODO: False value breaks backend logic. If this inf the same for all experiments, it should be calculated before experiment loop to avoid collisions
    capture_info = True

    worker_args: list[GenerateSingleHotspotReportParams] = []
    template_params = GenerateSingleHotspotReportParams(
        input_sql_filename=input_sql_filename,
        start_time_ns=start_time,
        end_time_ns=end_time,
        target_pid=target_pid,
        filtered_timeslice_list=filtered_timeslice_list,
        report_range=HotspotRange.SLOWEST,
        report_threading=HotspotReportThreading.BUSY_THREAD,
        report_metric=set(HotspotMetricType),
        report_frame_count=frame_count,
        report_thread_count=thread_count,
        report_median_frame=capture_median_frame,
        report_info=capture_info,
        report_output_path=report_output_path,
        report_exporter=report_exporter
    )

    # If the workload has CPU frames (ie. is a DX workload), look for stutter and periodics
    if cpu_frametimes_available:
        worker_args.append(template_params.copy_values(report_range=HotspotRange.PERIODIC, report_frame_count=periodic_frame_count, report_median_frame=False))
        worker_args.append(template_params.copy_values(report_range=HotspotRange.SLOWEST))
    else:  # else just look at the full trace
        worker_args.append(template_params.copy_values(
            report_range=HotspotRange.NONE,
            report_threading=HotspotReportThreading.BUSY_THREAD,
            report_frame_count=1,
            report_median_frame=False
        ))

    worker_args.append(template_params.copy_values(
        report_range=HotspotRange.CUDNN_GPU_KERNELS
    ))
    # Limit the metrics to GPU workload.
    worker_args.append(template_params.copy_values(
        report_range=HotspotRange.CUDNN_GPU_KERNELS,
        report_threading=HotspotReportThreading.ALL,
        report_metric={HotspotMetricType.NVTX_GPU_MARKERS, HotspotMetricType.CUDA_GPU_KERNELS},
        report_frame_count=0,
        report_median_frame=True
    ))
    worker_args.append(template_params.copy_values(
        report_range=HotspotRange.GR_IDLE,
        report_median_frame=False
    ))
    worker_args.append(template_params.copy_values(
        report_range=HotspotRange.PCIE_BAR1_READS,
        report_median_frame=False
    ))

    if multithreading:
        total_frame_count = sum([params.report_frame_count for params in worker_args], 0)
        progress.StartStep(total_frame_count, "Processing Hotspot Info")

    threads = []
    start = time.time()
    for params in worker_args:
        if multithreading:
            x = threading.Thread(target=generate_single_hotspot_report, args=[params])
            x.start()
            threads.append(x)
        else:
            progress.StartStep(params.report_frame_count, str(params.report_range))
            generate_single_hotspot_report(params)
    for thread in threads:
        thread.join()
    end = time.time()
    logger.info(f"Hotspot analysis completed in {end-start:.2f} seconds")
    cpu_frame_info.close_nsys_database()


def __escape_path(path: str):
    if ' ' in path and '"' not in path:
        return f'"{path}"'
    return path


def run_command(arg_list, base_path, timeout_s=None, shell=True):
    logger.info(f"Running command : {arg_list}; with timeout: {timeout_s}")
    escaped_arg_list = [__escape_path(arg) for arg in arg_list]
    logger.debug(f"Escaped command : {escaped_arg_list}")
    try:
        p = subprocess.check_output(' '.join(escaped_arg_list), cwd=base_path, stderr=subprocess.STDOUT, shell=shell,
                                    universal_newlines=True, timeout=timeout_s)
        logger.info(f"Output : [{p}]")
        return True
    except subprocess.CalledProcessError as e:
        logger.error(f"Failed to successfully execute command: {escaped_arg_list}")
        logger.error(f"\terror code: {e.returncode}")
        logger.error(f"\toutput: {e.output}")
        return False
    except subprocess.TimeoutExpired as e:
        logger.error(f"Timeout expired: {escaped_arg_list}")
        logger.error(f"\ttimeout: {e.timeout}")
        logger.error(f"\toutput: {e.output}")
        return False


def __get_nsys_version_numeric(nsys_binary):
    nsys_version_string = __get_nsys_version_string(nsys_binary)
    matches = re.search(r"(\d+).(\d+).(\d+)(.\S+)", nsys_version_string)
    if matches:
        return matches.group(1, 2)
    return None, None


def __get_nsys_version_string(nsys_binary):
    global g_nsys_version_string
    if g_nsys_version_string:
        return g_nsys_version_string
    subproc = subprocess.run([nsys_binary, "--version"], stderr=sys.stderr,
                             stdout=subprocess.PIPE, text=True)
    if 0 == subproc.returncode:
        g_nsys_version_string = subproc.stdout.strip()
        return g_nsys_version_string

    # Return a dummy value in the correct format in case of failure
    return r"NVIDIA Nsight Systems version UNKNOWN 2099.9.9"


def __get_nsys_report_version_string(filename):
    matches = re.findall(r"\[.*] (\d+.\d+.\d+.\S+)", filename)
    if len(matches) >= 1:
        return matches[0]
    return None


def __get_nsys_report_version_numeric(filename):
    matches = re.search(r"\[.*] (\d+).(\d+).(\d+)(.\S+)", filename)
    if matches:
        return matches.group(1, 2)
    return None, None


def __parse_path(file_path: str) -> tuple[str, str, str, str]:
    file_full_name = os.path.basename(file_path)
    file_base_name, file_extension = os.path.splitext(file_full_name)
    file_root_dir = os.path.dirname(os.path.abspath(file_path))
    return file_full_name, file_root_dir, file_base_name, file_extension


def validate_input_file_type(file_extensions: List[str]):
    def validate_function(value: str) -> str:
        if all((not value.endswith(f'.{file_extension}') for file_extension in file_extensions)):
            raise typer.BadParameter(f"Invalid input parameter : {value}. It should use {' or '.join(file_extensions)} file extension.")
        return value

    return validate_function


def __check_nsys_tool_version(nsys_binary, nsys_report_name):
    nsys_major, nsys_minor = __get_nsys_version_numeric(nsys_binary)
    nsys_version_string = __get_nsys_version_string(nsys_binary)
    logger.info(f"Using NSys version: {nsys_version_string}")

    need_newer_nsys = False
    nsys_report_major, nsys_report_minor = __get_nsys_report_version_numeric(nsys_report_name)

    if nsys_report_major and nsys_major and nsys_report_minor and nsys_minor:
        if nsys_report_major > nsys_major:
            need_newer_nsys = True
        elif nsys_report_major == nsys_major:
            if nsys_report_minor > nsys_minor:
                need_newer_nsys = True

    if need_newer_nsys:
        logger.error(f"Not processing file {nsys_report_name}")

        logger.error("\tThe report was captured with a newer version of nsys than is installed. Please update nsys")
        logger.error(f"\tInstalled NSys version: {nsys_version_string}; from here: {nsys_binary}")
        logger.error(f"\tReport NSys version: {__get_nsys_report_version_string(nsys_report_name)}")
        sys.exit(1)


def generate_sqlite(
        nsys_path: str = typer.Option(..., "--nsys_path", help="Root folder containing nsys installation"),
        input_file: str = typer.Option(..., "--input_file", help="Path to nsys report file",
                                       callback=validate_input_file_type(['nsys-rep', 'qdrep'])),
        output_path: str = typer.Option(..., "--output_path", help="Root folder for nsys report output"),
        output_sql_filename: str = typer.Option(..., "--output_sql_filename", help="SQL output filename"),
) -> Optional[str]:
    nsys_binary = __find_file_or_exit(g_nsys_cli_executable_name, nsys_path, g_nsys_sub_path)
    nsys_report_name, nsys_report_dir, nsys_report_base_name, nsys_report_extension = __parse_path(input_file)
    if not output_sql_filename or not isinstance(output_sql_filename, str):
        output_sql_filename = os.path.join(output_path, nsys_report_base_name + ".sqlite")

    if os.path.exists(output_sql_filename) and not g_do_sql_force:
        logger.debug("SKIPPING SQLite generation. File already exists.")
        return output_sql_filename

    # Only generate if it doesn't exist, or we are forcing it
    logger.info(f"Generating sqliteDB on file: {input_file}")
    start = time.time()
    generate_data = run_command([nsys_binary, "export", "--type=sqlite", "--force-overwrite=true", "--quiet=true",
                                 "--output=" + output_sql_filename, input_file], nsys_report_dir)

    if not generate_data:
        return None

    end = time.time()
    time_secs = end - start
    logger.info(f"SQLite generated in {time.strftime('%H:%M:%S', time.gmtime(time_secs))}")
    return output_sql_filename


def generate_report(
        nsys_sqlite_path: str = typer.Option(..., "--input_file", help="Path to sqlite report file", callback=validate_input_file_type(['sqlite'])),
        output_path: str = typer.Option(..., "--output_path", help="Root folder for output"),
        output_sqlite_name: str = typer.Option("nsys_cpu_stats.sqlite", "--output_sqlite_name", help="Filename for sqlite output"),
        start_time_sec: Optional[float] = typer.Option(None, "--start_time_sec", help="Start time for the report (seconds)."),
        end_time_sec: Optional[float] = typer.Option(None, "--end_time_sec", help="End time for the report (seconds)."),
        target_process_name: Optional[str] = typer.Option(None, "--target_process_name", help="Provide the target process name."),
):
    if g_clean_output_path:
        logger.info(f"Cleaning output dir: {output_path} ...")
        try:
            if os.path.exists(output_path):
                shutil.rmtree(output_path)
                os.mkdir(output_path)
                logger.debug(f"Directory '{output_path}' content removed successfully.")
            else:
                os.makedirs(output_path)
                logger.debug(f"Directory '{output_path}' was created.")
        except Exception as e:
            logger.error(f"Error removing directory '{output_path}': {e}")

    progress.StartStep(1, "Loading sqlite database")
    start_time, end_time, target_pid = find_target_pid_and_range(nsys_sqlite_path, target_process_name)
    if start_time_sec:
        logger.info(f"Over-riding start time with {start_time_sec}s.")
        start_time = start_time_sec * 1000000000
    if end_time_sec:
        logger.info(f"Over-riding end time with {start_time_sec}s.")
        end_time = end_time_sec * 1000000000

    report_name, _ = os.path.splitext(os.path.basename(nsys_sqlite_path))

    meta_info = tu.SourceMetaInfo(report_name=report_name)

    local_database_file = os.path.join(output_path, output_sqlite_name)
    report_exporter = ReportExporter(local_database_file=local_database_file)

    database_report_ids: Optional[DatabaseIds] = report_exporter.create_remote_report(meta_info)
    if database_report_ids:
        logger.info(f"Generated report remote ID: '{database_report_ids.remote}'")
        logger.info(f"Generated report local ID: '{database_report_ids.local}'")
        if database_report_ids.remote:
            with open(os.path.join(output_path, "report_id.txt"), 'w', encoding="utf-8") as output:
                output.write(str(database_report_ids.remote))

    # Capture the Threading report data
    progress.StartStep(1, "Processing Threading Info")
    if g_do_threading_info:
        logger.info("Threading Info processing start")
        generate_threading_info(nsys_sqlite_path, start_time, end_time, target_pid, output_path, meta_info, report_exporter)
        logger.info("Threading Info processing done")

    # Capture the Hotspot report data
    if g_do_hotspot_info:
        logger.info("Hotspot Info processing start")
        generate_hotspot_info(nsys_sqlite_path, start_time, end_time, target_pid, g_enable_multithreading, output_path, report_exporter)
        logger.info("Hotspot Info processing done")


@cli.command(help="Perform Analysis of Nsight Systems files.")
def process(
        nsys_dir_path: str = typer.Option(..., "--nsys_root_path", help="Root folder containing nsys installation"),
        input_file: str = typer.Option(..., "--input_file", help="Path to nsys report file", callback=validate_input_file_type(['nsys-rep', 'qdrep'])),
        output_path: str = typer.Option(..., "--output_path", help="Root folder for output"),
        output_sqlite_name: str = typer.Option("nsys_cpu_stats.sqlite", "--output_sqlite_name", help="Filename for sqlite output"),
        output_sqlite_path: Optional[str] = typer.Option(None, "--output_sqlite_path", help="Root folder for sqlite report output"),
        start_time_sec: Optional[float] = typer.Option(None, "--start_time_sec", help="Start time for the report (seconds)."),
        end_time_sec: Optional[float] = typer.Option(None, "--end_time_sec", help="End time for the report (seconds)."),
        target_process_name: Optional[str] = typer.Option(None, "--target_process_name", help="Provide the target process name."),
):
    total_start = time.time()
    if not output_sqlite_path:
        output_sqlite_path = output_path

    # isinstance() used to detect if the option has NOT been set, in which case it comes through as type typer.Option!
    if not isinstance(start_time_sec, float):
        start_time_sec = None
    if not isinstance(end_time_sec, float):
        end_time_sec = None
    if not isinstance(target_process_name, str):
        target_process_name = ""

    output_path = os.path.abspath(output_path)
    output_sqlite_path = os.path.abspath(output_sqlite_path)

    logger.info(f"NSys path = {nsys_dir_path}")
    logger.info(f"Input file = {input_file}")
    logger.info(f"Output path = {output_path}")
    logger.info(f"Output sqlite report path = {output_sqlite_path}")
    logger.info(f"Target Process Name = {target_process_name}")

    nsys_report_name, nsys_report_dir, file_name, file_extension = __parse_path(input_file)

    output_sql_filename = None

    progress.Initialize(
        1 + # SQLite export
        1 + # SQLite loading
        1 + # Threading Analysis
        1 if g_enable_multithreading else 7)  # Hotspot analysis reports

    # Check the Report and Nsys tools versions compatibility
    nsys_binary = __find_file_or_exit(g_nsys_cli_executable_name, nsys_dir_path, g_nsys_sub_path)
    __check_nsys_tool_version(nsys_binary, nsys_report_name)

    # Generate the SQL Lite
    progress.StartStep(1, "Exporting nsys-rep to sqlite")
    if g_do_sql:
        output_sql_filename = generate_sqlite(nsys_dir_path, input_file, output_sqlite_path)

    ##########################
    # Start Report generation
    ##########################
    if not output_sql_filename or not os.path.exists(output_sql_filename):
        nsys_version = __get_nsys_version_string(nsys_binary)
        logger.error(f"Failed to generate sql output file. Check version mismatch - this script is using nsys version: {nsys_version}")
    else:
        generate_report(nsys_sqlite_path=output_sql_filename,
                        output_path=output_path,
                        output_sqlite_name=output_sqlite_name,
                        start_time_sec=start_time_sec,
                        end_time_sec=end_time_sec,
                        target_process_name=target_process_name)

    ##########################
    # End Report generation
    ##########################
    progress.Finish()
    total_end = time.time()
    total_time_secs = total_end - total_start
    logger.info(f"Job completed in {time.strftime('%H:%M:%S', time.gmtime(total_time_secs))}")


@cli.command(help="Run the server application to view the report")
def run_viewer(
    report_path: str = typer.Option(..., "--report-path", help="Path to the nsys-cpu-stats sqlite report file"),
    server_port: int = typer.Option(8000, "--server-port", help="Server port")
):
    os.environ['NSYS_CPU_STATS_DATABASE_PROVIDER'] = 'sqlite'
    os.environ['NSYS_CPU_STATS_DATABASE_HOST'] = report_path
    # LLM functionality is not designed for local usage, so forcibly disable it
    os.environ['NSYS_CPU_STATS_STANDALONE'] = 'True'
    from server.app.main import app  # pylint: disable=import-outside-toplevel

    host_name = '127.0.0.1'

    config = uvicorn.Config(app=app, host=host_name, port=server_port, log_level="debug")
    server = uvicorn.Server(config=config)
    server.run()


if __name__ == '__main__':
    try:
        cli()
    except Exception as err:
        logger.error("Unhandled exception while processing CLI command")
        logger.error(err)
        logger.error(traceback.format_exc())
        sys.exit(1)
